# file: Router.rb
# note: source code is indented with tabs, tab-width=2
#
# A simple rubberband topological router based on the region concept mentioned in
# Tal Dayan's PhD thesis and other papers. 
# Currently we use random points as terminals and try to connect pairs of them.
# Currently we only generate a PGN picture (lines and arcs) using cairo.
#
# c Stefan Salewsk, mail@ssalewski.de
# License GPL
# Version 0.15 14-DEC-2013
#
# Call: ruby router.rb
# or ruby router.rb seed
# for reproduceable results
#
# ruby all.rb
# will route the t.pcb file, currently component side only
#
# Data type of coordinates
# ------------------------
# Generally integer data type should make our life easierer, but then we have to be careful
# when float datatype is mixed in by math operations. Special care is necessary when we use
# coordinates as Hash keys -- float and integer are of course different keys.
# Current default unit in PCB file format is 0.01 mil, generally in combination with integer
# numbers. Internal unit in PCB program is nm.
# For now we use 0.01 mil as our general coordinates. (For Ruby performance it may be a good idea
# to use only integers less than 2^30 in magnitude?)
#
# Markers used:
# TODO: needs improving
# UGLY: we should remove or at least carefully check this...
#
require 'set'
require_relative '../RBOOST/Fibonacci_Queue_Hash/rboost_fibonacci_queue'
require_relative '../RCGAL/rcgal_cdt'
require_relative '../RAPOLLONIUS/rcgal_apollonius'

require_relative 'rsupport'
require_relative 'canvas'

$glob = 0 

ARVS = 2000 # Autorouter Via Size -- later from config file

Board_Size = 800 # cairo PNG picture
PCB_Size = 14e4 # size of our set of random vertices
Points = 64 # terminal count for test

# defaults for testing -- of course later each pin/trace can have it's own values
# size unit is 0.01 mil, which is default unit of pcb program
Pin_Radius = 1000
#Trace_Width = 1200
#Trace_Width = 1200
Trace_Width = 1200
Clearance = 800
MinCutSize =6000

module RM

	def self.init_seed
		seed = (ARGV[0] ? ARGV[0].to_i : rand(1000))
		print 'random seed is ', seed, "\n"
		srand(seed)
		seed
	end

	# 	  b
	#    ^
	#   /
	# o/--------> a
	#
	def self.boolean_really_smart_cross_product_2d_with_offset(a, b, o)
		a = a.vertex if a.is_a? RBR::Region
		b = b.vertex if b.is_a? RBR::Region
		o = o.vertex if o.is_a? RBR::Region
		ax = a.x
		ay = a.y
		bx = b.x
		by = b.y
		ox = o.x
		oy = o.y
		return true if ax == bx && ay == by # TODO: undefined result, we should raise an exeption!
		ax -= ox
		ay -= oy
		bx -= ox
		by -= oy
		fail if (ax == 0 && ay == 0) || (bx == 0 && by == 0) # zero length should not occur
		if (p = ax * by - ay * bx) != 0
			p > 0
		else
			ax != bx ? ax < bx : ay < by # straight line -- ensure arbitrary but well defined result
		end
	end
end

module RBR # Rubber Band Router

# Net connecting two terminals (2Net)
class NetDesc
	@@id = 0
	attr_accessor :name1, :name2 # name of terminals
	attr_accessor :thickness
	attr_accessor :separation
	attr_accessor :id
	attr_accessor :pri # used for sorting by length -- poor mans net ordering
	attr_accessor :flag # used for sorting of attached nets
	def initialize(name1, name2, thickness = Trace_Width, separation = Clearance)
	  @id = @@id += 1
		@pri = 0
		@name1, @name2 = name1, name2
		@thickness, @separation = thickness, separation
	end
end

class NetDescList < Array
end

# Incident or attached net of a terminal
class Step
	attr_accessor :id # id of this net
	attr_accessor :net_desc
	attr_accessor :vertex # the terminal we are attached or incident to
	attr_accessor :prev, :next # terminal
	attr_accessor :pstep, :nstep # previous and next step
	attr_accessor :a, :b, :d, :g, :dir # for sorting, a few may be obsolete
	attr_accessor :radius # radius of arc
	attr_accessor :score # for sorting attached nets of a terminal by angle 
	attr_accessor :index # for sorting
	attr_accessor :ref # also for sorting
	attr_accessor :rgt # tangents of terminal for this step -- left or right side in forward direction
	attr_accessor :outer # do we use the outer lane at this terminal
	attr_accessor :xt # do tangents cross each other -- if so we can collapse this concave step
	attr_accessor :lr_turn # left or right turn

	def initialize(prev, nxt, id)
		@prev, @next, @id = prev, nxt, id
		@radius = 0 # default for incident net
		@outer = false
	end
end

class Tex # plain (temporary) terminal (verTex)
	attr_accessor :x, :y
	def initialize(x, y)
		@x, @y = x, y
	end
end

# Terminal
class Vertex < CGAL::Vertex
	@@id = @@cid = 0
	attr_accessor :id # general unique id number
	attr_accessor :cid # cluster id; -1 for plain pins, nonnegativ values if vertex is part of a pad/cluster
	attr_accessor :vis_flag # for debugging, graphical color mark
	attr_accessor :core # outer copper of pin itself, without attached traces
	attr_accessor :radius # outer copper of outermost attached trace, equal to core wnen no attached nets exist
	attr_accessor :separation # outer clearance of pin itself or of outermost attached trace
	attr_accessor :neighbors # Vertex/Terminal, the neighbors in the delaunay triangulation
	attr_accessor :incident_nets # Step, nets incident to this terminal
	attr_accessor :attached_nets # Step, nets attached to this terminal
	attr_accessor :name # name of the Terminal, i.e. U7_3 -- pin 3 of IC U7
	attr_accessor :tradius, :trgt # temporary data
	attr_accessor :outer # temporary data for routing,
	attr_accessor :lr_turn # later copied to step
	attr_accessor :via # vertex is a via

	def initialize(x = 0, y = 0, r = Pin_Radius, c = Clearance)
		super(x, y)
		@via = false
		@tradius = 0
		@vis_flag = 0
	  @id = @@id
		@cid = -1
		@@id += 1
		@radius = @core = r
		@separation = c
		#@core = r
		@name = ''
		@neighbors = Array.new
		@incident_nets = Array.new
		@attached_nets = Array.new
	end

	def self.begin_new_cluster # class method!
		@@cid += 1
	end

	def add_to_current_cluster
		@cid = @@cid
	end

	def xy
		return x, y
	end

  def reset_initial_size
		@radius, @separation = @core, Clearance
	end

	# UGLY: may be obsolete -- at least it is only an estimation
	def resize
		reset_initial_size
		attached_nets.each{|step|
			net = step.net_desc
			trace_sep = [@separation, net.separation].max
			@radius += trace_sep + net.thickness
			step.radius = @radius - net.thickness * 0.5
			@separation = net.separation
		}
	end

	# UGLY: may be obsolete -- at least it is only an estimation
  def update(s)
		net = s.net_desc
		trace_sep = [@separation, net.separation].max
		@radius += trace_sep + net.thickness
		s.radius = @radius - net.thickness * 0.5
		@separation = net.separation
	end

	# UGLY: returns step -- may become obsolete
  def net(id)
		incident_nets.each{|s| return s if s.id == id}
 		attached_nets.each{|s| return s if s.id == id}
		return nil
	end

  # UGLY: delete step -- currently we need this, but we should try to avoid it, at least the expensive resize operation
	def new_delete_net(step)
		incident_nets.delete_if{|s| step == s}
		attached_nets.delete_if{|s| step == s}
		resize
	end

	# UGLY:
	def _full_angle(s)
		return nil unless s.next && s.prev
		v = s.vertex
		d = Math.atan2(s.next.y - v.y, s.next.x - v.x) - Math.atan2(v.y - s.prev.y, v.x - s.prev.x)
		if d < -Math::PI
		 	d += 2 * Math::PI
		elsif d > Math::PI
			d -= 2 * Math::PI
		end
		return d
	end

	# UGLY: check and improve
	def sort_attached_nets # by angle of total trace
		unless attached_nets.length < 2
			attached_nets.each{|n|
				fail unless n.vertex == self
				#n.index = _tangents_full_angle(n) # we may need the tangents angle?
				n.index = _full_angle(n) * (n.rgt ? 1 : -1)
			}
			attached_nets.sort_by!{|n| n.index}
			attached_nets.each_with_index{|n, i| n.index = i}
			shash = Hash.new
			attached_nets.each{|n| # group attached nets with same angle (overlapping)
				l = n.prev
				r = n.next
				n.net_desc.flag = 1
				if shash.has_key?([l, r])
					shash[[l, r]] << n
				elsif shash.has_key?([r, l])
					n.net_desc.flag = -1 # inverted direction
					shash[[r, l]] << n
				else
					shash[[l, r]] = [n]
				end
			}
			shash.each_value{|group| # fine sort each group by following the traces
				if group.length > 1
					group.reverse! # for testing -- initialy reversing the group should give same result!
					group.each{|el| el.ref = el}
					indices = Array.new
					group.each{|el| indices << el.index}
					indices.sort!
					rel = Hash.new
					[-1, 1].each{|direction|
						gr = group.dup
						final = true # for first direction we may get only a preliminary order?
						while gr.length > 1
							gr.map!{|el| (el.net_desc.flag == direction ? el.pstep : el.nstep)} # walk in one direction
							gr.each{|el| el.ref = (el.net_desc.flag == direction ? el.nstep.ref : el.pstep.ref)}
							gr.each{|el| el.score = _full_angle(el)}
							unresolved_combinations = false
							gr.combination(2).each{|el|
								a, b = *el
								relation = rel[[a.ref, b.ref]]
								if !relation || relation.abs < 2
									if !a.score
										c = ((b.rgt == b.ref.rgt) ? 1 : -1)
									elsif !b.score
										c = ((a.rgt == a.ref.rgt) ? -1 : 1)
									else
										if (a.score * a.net_desc.flag - b.score * b.net_desc.flag).abs < 1e-6
										#if ((a.score - b.score).abs < 1e-6) || ((a.score - b.score).abs < 1e-6)
											c = 0
										else
											#c = ((a.score * (a.rgt ? 1 : -1) * ((a.rgt == a.ref.rgt) ? 1 : -1)) <=> (b.score * (b.rgt ? 1 : -1) * ((b.rgt == b.ref.rgt) ? 1 : -1)))
											c = ((a.score * (a.ref.rgt ? 1 : -1)) <=> (b.score * (b.ref.rgt ? 1 : -1))) # same as above
										end
									end
									if c != 0
										if  final # indicate final relation
											c *= 2
										end
											rel[[a.ref, b.ref]] = c
											rel[[b.ref, a.ref]] = -c
									else
										unresolved_combinations = true
									end
								end
							}
							break unless unresolved_combinations
							gr.keep_if{|el| el.next && el.prev}
						end
						fail if unresolved_combinations # we should get at least a preliminary relation
						break if final # indeed always -- we have no preliminary relations
					}
					group.sort!{|a, b| rel[[a, b]]} # do we need rel[[a, b] <=> 0 to ensure -1,0,1 in block? 
					group.each{|el| el.index = indices.shift}
				end
			}
			attached_nets.sort_by!{|el| -el.index}
		end
	end
end

#   \|/       \|/
#    |        /\
#    |        | |
#   _|/      _| |/ 
#    |\       | |\
# when we route a net, we split regions along the path in left and right region.
# so new paths can not cross this path any more.
# problem: first and last terminal.
# Current solution: Use a hash to store forbidden paths for these terminals
# 
class Region
	attr_accessor :vertex, :neighbors, :incident, :outer, :restricted_pairs
	attr_accessor :g, :ox, :oy
	attr_accessor :a # for sorting
	attr_accessor :lr_turn

  def initialize(v = nil)
		@g = 1
		@ox = @oy = 0
		@vertex = v
		@neighbors = Array.new
		@restricted_pairs = Set.new
		#@name = nil
		@incident = true
		#@outer_allowed = true
		@outer = false
	end

	def qbors(old)
		@neighbors.each{|el|
			only_outer = old && (@restricted_pairs.include?([old, el]) || @restricted_pairs.include?([el, old]) || 
			(old.vertex.x == el.vertex.x && old.vertex.y == el.vertex.y))
			yield [el, only_outer]
		}
	end

	def distance_to(other)
		Math.hypot(@vertex.x - other.vertex.x, @vertex.y - other.vertex.y)
	end
end

class Cut
  attr_accessor :cap, :flow
  def initialize(cap, flow)
		@cap = cap
		@flow = flow
	end
end

# we put only one pin in each cell -- for testing
CS = 3 * Pin_Radius

class Router
	attr_accessor :filename, :netlist
	attr_accessor :file # pcb_layout_data_file
	attr_accessor :edges_in_cluster # diagonal edge in pad/cluster, forbidden path, we may consider removing it from triagulation
  def initialize(b1x, b1y, b2x, b2y)
		@b1x, @b1y, @b2x, @b2y= b1x, b1y, b2x, b2y # corners of the PCB board
		@edges_in_cluster = RouterSupport::Hash_with_ordered_array_index.new
		@name_id = 0
	  @path_ID = 0
		@image = Cairo::ImageSurface.new(Board_Size, Board_Size)
		@pic = Canvas::Pic.new(@image)
		@pic.translate(40, 40)
		@pic.scale(0.9, 0.9)
		x_extent = (b2x - b1x).abs
		y_extent = (b2y - b1y).abs
		max_extent = [x_extent, y_extent].max
		@pic.scale(Board_Size.fdiv(max_extent), Board_Size.fdiv(max_extent))
		@pic.translate(-b1x, -b1y)
		@pic.translate((max_extent - x_extent) * 0.5, (max_extent - y_extent) * 0.5)
		@pic.set_source_rgba(0.8, 0.8, 0.8, 1)
		@pic.paint
		@cdt = CGAL::CDT.new
		@cdt_hash = Hash.new
		@cell = Hash.new
	end

	def next_name
		@name_id += 1
		return @name_id.to_s
	end

	# UGLY:
	# insert some vertices on the boarder of the PCB board, should help
	def insert_pcb_border
		a, b = [@b1x, @b2x].minmax
		d = (b - a).fdiv(25)
		a -= d
		b += d
		dx = (b - a).fdiv(10)
		(a..b).step(dx){|x|
			v = Vertex.new(x, @b1y - d)
			v.name = 'no'
			@cdt.insert(v)
			v = Vertex.new(x, @b2y + d)
			v.name = 'no'
			@cdt.insert(v)
		}
		a, b = [@b1y, @b2y].minmax
		d = (b - a).fdiv(25)
		a -= d
		b += d
		dy = (b - a).fdiv(10)
		a += dy
		b -= dy
		(a..b).step(dy){|y|
			v = Vertex.new(@b1x - d, y)
			v.name = 'no'
			@cdt.insert(v)
			v = Vertex.new(@b2x + d, y)
			v.name = 'no'
			@cdt.insert(v)
		}
	end

	# UGLY:
	def insert_pcb_vertex(name, x, y)
		return if @cdt_hash.include?([x, y]) # occurs for t.pcb
		v = Vertex.new(x, y, 2 *Pin_Radius)
		v.via = true
		v.name = name
		@cdt_hash[[x, y]] = v
		@cdt.insert(v)
	end

	def insert_cluster(c)
		fail if c.convex_pin_hull.empty?
		Vertex.begin_new_cluster unless (n = c.convex_pin_hull.size) == 1
		last_vertex = first_vertex = nil
		c.convex_pin_hull.each{|cv|
			x = c.mx + cv.rx
			y = c.my + cv.ry
			if @cdt_hash.include?([x, y])
				fail
			else
				v = Vertex.new(x, y, cv.thickness * 0.5, cv.clearance)
				v.name = c.name
				v.add_to_current_cluster unless n == 1
				first_vertex ||= v
				@cdt_hash[[x, y]] = v
				@cdt.insert(v)
				@cdt.insert_constraint(v, last_vertex) if last_vertex
				last_vertex = v
			end
		}
		@cdt.insert_constraint(last_vertex, first_vertex) if n > 2
	end

	# UGLY: rename
	def test_cluster
		@cdt.edges_in_constrained_polygons{|v1, v2|
			@edges_in_cluster[v1, v2] = true
		}
	end

	# UGLY: 
  def generate_test_vertices
		# put a virtual pin on each corner of our board -- we should need this
		[0, PCB_Size].repeated_permutation(2).each{|el|
			v = Vertex.new(*el)
			@cdt.insert(v)
			@cell[[el[0] / CS, el[1] / CS]] = 1
		}
		id = 8
		while id < Points
			r1, r2 = rand(PCB_Size - PCB_Size / 50) + PCB_Size / 100, rand(PCB_Size - PCB_Size / 50) + PCB_Size / 100
			if @cell[[r1 / CS, r2 / CS]] == nil
				@cell[[r1 / CS, r2 / CS]] = 1
				#if true#unless (300..500).include?(r1) or (300..500).include?(r2)
				v = Vertex.new(r1, r2)
				@cdt.insert(v)
				id += 1
			end
		end

	end

	def generate_netlist(l)
		@netlist = NetDescList.new
		l.each{|x1, y1, x2, y2|
			v1 = @cdt_hash[[x1, y1]]
			v2 = @cdt_hash[[x2, y2]]
			fail unless v1 && v2
			v1.name ||= next_name
			v2.name ||= next_name
			net_desc = NetDesc.new(v1.name, v2.name)
			net_desc.pri = (x2 - x1) ** 2 + (y2 - y1) ** 2
			@netlist << net_desc
		}
	end

	def sort_netlist
		@netlist.sort_by!{|el| el.pri} 
	end

  def finish_init(rnd_test = false)
		@vertices = Array.new # maybe later we do not need this array, we have @cdt.each{}
		@regions = Array.new
		@cdt.each{|v|
			@vertices << v
			@regions << Region.new(v)
		}
		if rnd_test
			set = @vertices.select{|el| el.id > 3}.each_slice(2).to_a #.shuffle
			set.sort_by!{|el| (el[0].x - el[1].x) ** 2 + (el[0].y - el[1].y) ** 2}
			@netlist = NetDescList.new
			(0..9).each{|i|
				v1, v2 = *set[i * 3]
				if v1 && v2
					v1.name = i.to_s + 's'
					v2.name = i.to_s + 'e'
					net_desc = NetDesc.new(v1.name, v2.name)
					@netlist << net_desc
				end
			}
		end
		@newcuts = RouterSupport::Hash_with_ordered_array_index.new
		@cdt.each{|v|
			@cdt.neighbor_vertices(v).each{|n|
				v.neighbors << n
				if @regions[v.id]
					@regions[v.id].neighbors << @regions[n.id]
				end
				#@newcuts[v, n] = Cut.new(Math::hypot(v.x - n.x, v.y - n.y), 4200)# - v.radius - n.radius)
				@newcuts[v, n] = Cut.new(Math::hypot(v.x - n.x, v.y - n.y), v.core + v.separation + n.core + n.separation)
				fail unless v.core == v.radius

				#puts 'bups', v.core, v.separation

				#end
				#print '----------', v.radius, ' ', n.radius, "\n"
			}
		}
	end

	# UGLY: for debugging only
	def flag_vertices
	@pic.set_font_size(4 * Pin_Radius)
		@pic.set_source_rgba(1, 0, 0, 1)
		@pic.set_line_width(1)
		@vertices.each{|v|
		@pic.move_to(v.x, v.y)
		#@pic.show_text(v.id.to_s)
			if v.vis_flag != 0
				if v.vis_flag == 1
					@pic.set_source_rgba(1, 1, 1, 1)
				else
					@pic.set_source_rgba(0, 1, 0, 1)
				end
				@pic.new_sub_path
				@pic.set_line_width(1)
				@pic.arc(v.x, v.y, 2* Pin_Radius, 0, 2*Math::PI)
				@pic.fill
				@pic.stroke
			end
		}
	end

	# UGLY:
	def draw_vertices
		@pic.set_source_rgba(0, 0, 0, 0.3)
		@pic.set_line_width(100)
		@vertices.each{|v|
			@pic.new_sub_path
			if v.cid == -1
			@pic.arc(v.x, v.y, v.core, 0, 2 * Math::PI)
			else
			@pic.arc(v.x, v.y, Pin_Radius, 0, 2 * Math::PI)
			end
			@pic.fill
			@pic.stroke
			v.neighbors.each{|n|
			if @edges_in_cluster.include?(v, n)
		@pic.set_source_rgba(1, 0, 0, 0.3)
				@pic.move_to(v.x, v.y)
				@pic.line_to(n.x, n.y)
				@pic.stroke
			else
		@pic.set_source_rgba(0, 0, 0, 0.3)
				@pic.move_to(v.x, v.y)
				@pic.line_to(n.x, n.y)
				@pic.stroke
			end
			}
		}
		@pic.stroke
		@pic.set_source_rgba(1, 0, 0, 0.7)
		@pic.set_line_width(600)
		@newcuts.each_pair{|k, v|
		#unless v.cid >= 0 && v.cid == k.cid
			if v.cap < MinCutSize
				#@pic.move_to(k[0].x, k[0].y)
				#@pic.line_to(k[1].x, k[1].y)
			end
		#end
		}
		@pic.stroke
	end


	def gen_vias
#Via[95400 167800 4000 2000 0 2000 "" ""]
@vertices.each{|v|
if v.via
#@file.write("Via[95400 167800 4000 2000 0 2000 \"\" \"\"]\n")

@file.write("Via[#{v.x.round} #{v.y.round} #{(2 * v.core).round} #{Clearance.round} #{0} #{1000} \"\" \"\"]\n")


end
}
	end

	def gen_line(x0, y0, x1, y1, w)
		@pic.set_line_width(w)
		@pic.move_to(x0, y0)
		@pic.line_to(x1, y1)
		@pic.stroke
#	Line[257500 222500 257500 237500 1000 2000 ""]
@file.write("Line[#{x0.round} #{y0.round} #{x1.round} #{y1.round} #{w.round} #{Clearance.round} \"\"]\n")

	end

  def gen_arc(x, y, r, from, to, width)
		@pic.new_sub_path
		@pic.set_line_width(width)
		@pic.arc(x, y, r, from, to)
		@pic.stroke
	#Arc[192900 187300 11700 11700 1000 2000 -90 -90 ""]

to += 2 * Math::PI if to < from
#to += 2 * Math::PI if to < from
#to += 2 * Math::PI if to < from

puts "alert" if to - from > 2


#pcb_start_angle = ((from > Math::PI ? from - Math::PI : from + Math::PI) * 180 / Math::PI).round
pcb_start_angle = Math::PI - from
pcb_start_angle += 2 * Math::PI if pcb_start_angle < 0
pcb_start_angle -= 2 * Math::PI if pcb_start_angle > 2 * Math::PI
pcb_start_angle = (pcb_start_angle * 180 / Math::PI).round


pcb_end_angle = Math::PI - to
pcb_end_angle += 2 * Math::PI if pcb_end_angle < 0
pcb_end_angle -= 2 * Math::PI if pcb_end_angle > 2 * Math::PI
pcb_end_angle = (pcb_end_angle * 180 / Math::PI).round







#pcb_start_angle = (from + Math::PI * 180 / Math::PI).round
if from < to
pcb_delta_angle = ((from - to) * 180 / Math::PI).round
else
pcb_delta_angle = -((from - to) * 180 / Math::PI).round
end
pcb_delta_angle = (to - from)

puts "pcbalert" if pcb_delta_angle > 2
pcb_delta_angle = -(pcb_delta_angle * 180 / Math::PI).round


unless pcb_delta_angle == 0
@file.write("Arc[#{x.round} #{y.round} #{r.round} #{r.round} #{width.round} #{Clearance.round} #{pcb_start_angle} #{pcb_delta_angle} \"\"]\n")
end
	end

# http://www.faqs.org/faqs/graphics/algorithms-faq/
# http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html
# int pnpoly(int nvert, float *vertx, float *verty, float testx, float testy)
# {
#   int i, j, c = 0;
#   for (i = 0, j = nvert-1; i < nvert; j = i++) {
#     if ( ((verty[i]>testy) != (verty[j]>testy)) &&
# 	 (testx < (vertx[j]-vertx[i]) * (testy-verty[i]) / (verty[j]-verty[i]) + vertx[i]) )
#        c = !c;
#   }
#   return c;
# }
#
# input: array of vertices
# result: the vertices inside the polygon (or on the border?)
	def vertices_in_polygon(p_vertices, test_vertices)
		res = Array.new
		nm1 = p_vertices.length - 1
		test_vertices.each{|tp|
			ty = tp.y
			i = 0
			j = nm1
			c = false
			while i <= nm1
        if ((((p_vertices[i].y <= ty) && (ty < p_vertices[j].y)) ||
             ((p_vertices[j].y <= ty) && (ty < p_vertices[i].y))) &&
            (tp.x < (p_vertices[j].x - p_vertices[i].x) * (ty - p_vertices[i].y) / (p_vertices[j].y - p_vertices[i].y) + p_vertices[i].x))
					 c = !c
				end
				j = i
				i += 1
			end
			res << tp if c
		}
		res
	end

	#     (x2,y2)
	#    /
	#   /    (x0,y0)
	#  /
	# (x1,y1)
	# http://mathworld.wolfram.com/Point-LineDistance2-Dimensional.html
	def distance_line_point(x1, y1, x2, y2, x0, y0)
		x12 = x2 - x1
		y12 = y2 - y1
		(x12 * (y1 - y0) - (x1 - x0) * y12).abs / Math.hypot(x12, y12)
	end

	def distance_line_point_squared(x1, y1, x2, y2, x0, y0)
		x12 = x2 - x1
		y12 = y2 - y1
		(x12 * (y1 - y0) - (x1 - x0) * y12) ** 2 / (x12 ** 2 + y12 ** 2)
	end

	#      (c)
	#     /
	#    /     (p)
	#   /
	# (b)
	# see http://www.geometrictools.com/
	# see also http://paulbourke.net/geometry/pointlineplane/
	#
	def unused_distance_line_segment_point_squared(bx, by, cx, cy, px, py)
		mx = cx - bx
		my = cy - by
		hx = px - bx
		hy = py - by
		t0 = (mx * hx + my * hy).fdiv(mx ** 2 + my ** 2)
		if t0 <= 0
		elsif t0 < 1
			hx -= t0 * mx
			hy -= t0 * my
		else
			hx -= mx
			hy -= my
		end
		return hx ** 2 + hy ** 2
	end

	# Intersection point of two lines in 2 dimensions
	# http://paulbourke.net/geometry/pointlineplane/
	def line_line_intersection(x1, y1, x2, y2, x3, y3, x4, y4)
		x2x1 = x2 - x1
		y2y1 = y2 - y1
		return nil if (d = (y4 - y3) * x2x1 - (x4 - x3) * y2y1) == 0 # parallel?
		ua = ((x4 - x3) * (y1 - y3) - (y4 - y3) * (x1 - x3)) / d
		ub = (x2x1 * (y1 - y3) - y2y1 * (x1 - x3)) / d
		[x1 + ua * x2x1, y1 + ua * y2y1, ua, ub]
	end

	# P_IN = 1; P_ON = 0; P_OUT = -1 
	# see http://en.wikipedia.org/wiki/Barycentric_coordinates_%28mathematics%29
	def unused_point_in_triangle(x1, y1, x2, y2, x3, y3, x, y)
		d  =  (y2 - y3) * (x1 - x3) + (x3 - x2) * (y1 - y3)
		l1 = ((y2 - y3) * (x - x3) + (x3 - x2) * (y - y3)) / d
		l2 = ((y3 - y1) * (x - x3) + (x1 - x3) * (y - y3)) / d
		l3 = 1 - l1 - l2
		min, max = [l1, l2, l3].minmax
		if 0 <= min && max <= 1
			0 < min && max < 1 ? P_IN : P_ON
		else
			P_OUT
		end
	end

	def unused_vertices_in_triangle(x1, y1, x2, y2, x3, y3, vertices)
		d  =  (y2 - y3) * (x1 - x3) + (x3 - x2) * (y1 - y3)
		v_in = Array.new
		if d == 0
			puts 'vertices_in_triangle, degenerated'
			return v_in
		end
		y2my3 = (y2 - y3) / d
		x3mx2 = (x3 - x2) / d
		x1mx3 = (x1 - x3) / d
		y3my1 = (y3 - y1) / d
		vertices.each{|v|
			vxmx3 = v.x - x3
			vymy3 = v.y - y3
			l1 = y2my3 * vxmx3 + x3mx2 * vymy3
			l2 = y3my1 * vxmx3 + x1mx3 * vymy3
			min, max = [l1, l2, 1 - l1 - l2].minmax
			if 0 <= min && max <= 1
				v_in << v
			end
		}
		v_in
	end

	# see http://en.wikibooks.org/wiki/Algorithm_Implementation/Geometry/Convex_hull/Monotone_chain
	def unused_hcross(o, a, b)
		(a[0] - o[0]) * (b[1] - o[1]) - (a[1] - o[1]) * (b[0] - o[0])
	end
	def unused_convex_hull(points)
		points.sort!.uniq!
		return points if points.length < 3
		lower = Array.new
		points.each{|p|
			while lower.length > 1 and unused_hcross(lower[-2], lower[-1], p) <= 0 do lower.pop end
			lower.push(p)
		}
		upper = Array.new
		points.reverse_each{|p|
			while upper.length > 1 and unused_hcross(upper[-2], upper[-1], p) <= 0 do upper.pop end
			upper.push(p)
		}
		return lower[0...-1] + upper[0...-1]
	end

	def new_convex_vertices(vertices, prev, nxt, hv1, hv2)
		fail if vertices.include?(prev) || vertices.include?(nxt)
		return vertices if vertices.empty?
		x1, y1, x2, y2 = get_tangents(prev.x, prev.y, prev.tradius, prev.trgt, nxt.x, nxt.y, nxt.tradius, nxt.trgt)
		v1 = Vertex.new(x1, y1)
		v2 = Vertex.new(x2, y2)
		vertices << v1 << v2 << hv1 << hv2
		ag = CGAL::Apollonius_graph.new
		vertices.each{|v| ag.insert(v, v.x, v.y, v.tradius)}
		x2 -= x1
		y2 -= y1
		(ag.convex_hull - [v1, v2, hv1, hv2]).sort_by{|el| (el.x - x1) * x2 + (el.y - y1) * y2}
	end

	# https://en.wikipedia.org/wiki/Dijkstra%27s_algorithm
	# u --> v --> w
	# Dijkstra's shortest path search -- along the edges of the constrained delaunay triangulation
	#
	#        v-----w
	#  \    / \
	#   \  /   \ lcut
	#    u      x
	#
	# Generally we route at the inner lane -- doing so ensures that traces never cross.
	# When the turn at u is opposite to the turn at v then we have to cross
	# the line segment connecting u and v -- we need space for that.
	# When we take the inner lane at v, then the lcut to vertex x is a restiction. 
	# When there starts an incident net in the inner lane this path
	# is blocked, so we can use the outer lane, there can exist no inner
	# crossing lanes. If we take the outer lane at v and inner lane at u,
	# we have not to cross the line segment connection u and v and so on...
	#
	# So for this variant of Dijkstra's shortest path search we have not only to
	# record the previous node, but also the lane (inner/outer) we took, because
	# the next node may be only reachable when we do not have to cross a line
	# segment between current and next node. We also record the parent of the previous
	# node (u) -- because we have to check if there exists a terminal too close to u-v.
	#
	# So the key for the Fibonacci_Queue and parents and distances is not simple
	# a node, but a tripel of node, previous node and lane.
	#
	# Default unit of Size is 0.01 mil
	# We will put this into a config file later
	MAX_PCB_BOARD_DIA = 2 ** 30 # max board diagonal == max length that can occur
	AVERAGE_VIA_SIZE = 3000

	#	TODO: Check backside of first and last vertex of each path -- i.e. when traces
	#	are thicker than pad or pin. Maybe we should leave that for manually fixing.
	#	Ckecking is not really easy, and automatic fixing is some work.

	def dijkstra(start_node, end_node_name)
		fail unless start_node.is_a? Region
		fail unless end_node_name.is_a? String
		fail if end_node_name.empty?
		fail if start_node.vertex.name == end_node_name
		q = BOOST::Fibonacci_Queue.new(-1, Float::INFINITY) # -1 for minimum queue
		distances = Hash.new
		parents = Hash.new
		outer_lane = Hash.new # record if we used inner or outer trail
		distances[[start_node, nil, true]] = 0 # fake left and right turn before start node
		distances[[start_node, nil, false]] = 0
		x, y = start_node.vertex.xy
		start_cid = start_node.vertex.cid # cluster id, -1 means no pad/cluster but plain pin
		start_node.qbors(nil) do |w, only_outer| # initial steps are never blocked, so fill them in
			u = [w, start_node, false] # for rgt == true and rgt == false, so we can continue in all directions
			v = [w, start_node, true]
			q[u] = q[v]	= ((start_cid != -1) && (w.vertex.cid == start_cid) ? 0 : Math.hypot(w.vertex.x - x, w.vertex.y - y))
			parents[u] = parents[v] = [start_node, nil, false] # arbitrary u and rgt for last two array elements
		end
		while true do
			min, old_distance = q.pop
			return nil unless min
			fail unless min.length == 3
			$glob += 1
			v, uu, prev_rgt = *min
			fail unless v && uu
			hhh = (uu == start_node || (uu.vertex.cid == start_cid && start_cid != -1) ? 0 : uu.vertex.radius)
			if (v.vertex.name == end_node_name) && v.incident # reached destination -- check if we touched a vertex
				hhh	= get_tangents(*uu.vertex.xy, hhh, prev_rgt, *v.vertex.xy, 0, false) # last two arguments are arbitrary
				(uu.vertex.neighbors & v.vertex.neighbors).each{|el| # only two -- maybe CGAL query is faster?
					next if distance_line_point(*hhh, el.x, el.y) < 1.5 * el.radius # TODO: Fix exact radius
				}
				# for now avoid long twisted paths which may block others
				return nil if old_distance > 2 * Math::hypot(v.vertex.x - start_node.vertex.x, v.vertex.y - start_node.vertex.y)
				break
			end
			vcid = v.vertex.cid
			distances[min] = old_distance
			x, y = v.vertex.xy
			pom = parents[min] # parent of min
			u = pom[0]
			fail unless u == uu
			path = Set.new # prevent loops
			p = min
			while p
				path << p[0]
				p = parents[p]
			end
			uv_blocked = [false, true].map{|b| # does path u-v touch a vertex if we use inner/outer lane at v? [cur_rgt==false, cur_rgt==true]
				p	= get_tangents(*uu.vertex.xy, hhh, prev_rgt, *v.vertex.xy, v.vertex.radius, b)
				(uu.vertex.neighbors & v.vertex.neighbors).each{|el|
					break true if distance_line_point(*p, el.x, el.y) < el.radius # TODO: Fix exact radius
				}
				false
			}
			v.qbors(u) do |w, only_outer|
				next if @edges_in_cluster.include?(v.vertex, w.vertex) # diagonal edge in pad/cluster
				hhh = false
				if false #u == w && u != start_node # direct connection of two adjanced vertices, allow 2 PI loop around
				# indeed this can currently not work: we store lr_turn and outer attributes in region for each step, but do not copy region! 
					v.vertex.incident_nets.map{|el| el.nstep || el.pstep}.each{|el|
						hhh = true if !(el.nstep && el.pstep) && (el.vertex == u.vertex)
					}
				else
					next if path.include?(w)
				end
				if false # we ignore this for now -- complicated and not really necessary...
				#if hhh && pom[1] != v && pom[1] != w && pom[1].vertex != v.vertex && pom[1].vertex != w.vertex
					only_outer = true
					lr_turn = RM.boolean_really_smart_cross_product_2d_with_offset(pom[1], w, v)
				elsif u.vertex == w.vertex # 2 PI turn -- determine left/right turn by center of mass of neightbors
					fail unless only_outer
					cm = [u.neighbors, w.neighbors].map{|el|
						Tex.new(el.inject(0){|sum, n| sum + n.vertex.x}.fdiv(el.length), el.inject(0){|sum, n| sum + n.vertex.y}.fdiv(el.length))
					}
					lr_turn = RM.boolean_really_smart_cross_product_2d_with_offset(cm[0], cm[1], v.vertex) # left or right turn?
				else
					lr_turn = RM.boolean_really_smart_cross_product_2d_with_offset(u, w, v) # left or right turn?
				end
				cur_rgt = lr_turn # lr_turn ^ outer
				w_v_rgt = [w, v, cur_rgt]
				w_v_xrgt = [w, v, !cur_rgt]
				if (start_cid != -1) && (vcid == start_cid) && (w.vertex.cid == start_cid) # free walk around boundary of start cluster
					[w_v_rgt, w_v_xrgt].each{|el|
						unless distances.include?(el)
							if q.inc?(el, old_distance)
								parents[el] = min
							end
						end
					}
					next
				end
				next if only_outer && vcid > -1 # corner of cluster, inner path not allowed, outer also forbidden!
				new_distance = old_distance + Math.hypot(w.vertex.x - x, w.vertex.y - y)
				outer_distance = new_distance # save this value
				lcuts = nil
				if !(can_out = only_outer) && !distances.include?(w_v_rgt)
					not_blocked = catch(:blocked){
						# process test with <can_out = true> clauses first!
						lcuts = new_bor_list(u, w, v) # neighbours in the inner angle/lane
						if vcid >= 0 # at corner of pad/cluster
							if lcuts.find{|el| el.cid == vcid} || (u.vertex.cid == vcid && w.vertex.cid == vcid && lcuts.empty?)
								can_out = true
								throw :blocked
							end
						end
						# now test special case when we touch a terminal with incident nets
						v.vertex.incident_nets.map{|el| el.nstep || el.pstep}.each{|el|
							hhh = lcuts.include?(el.vertex)
							can_out = true if hhh && (vcid == -1)
							throw :blocked if hhh
							throw :blocked if !(el.nstep && el.pstep) && (cur_rgt != prev_rgt) && (el.vertex == u.vertex)
						}
						throw :blocked if uv_blocked[(cur_rgt ? 1 : 0)]
						#hhh = @newcuts[v.vertex, el]
						lcap = lcuts.inject(MAX_PCB_BOARD_DIA){|min, el|
							hhh = @newcuts[v.vertex, el]
							[hhh.cap - hhh.flow, min].min
						}
						throw :blocked if lcap < Trace_Width
						hhh = @newcuts[v.vertex, u.vertex]
						throw :blocked if (u != start_node) && (cur_rgt != prev_rgt) && hhh.cap - hhh.flow < Trace_Width
						if hhh = @newcuts[w.vertex, u.vertex] # looking for shorter paths
							nd = distances[pom] + hhh.cap * 1.1
							if nd < new_distance
								new_distance = [nd, old_distance].max
							end
						elsif lcap * 2 > v.distance_to(w) + v.distance_to(u) # good chances for collapse
							nd =  distances[pom] + Math.hypot(u.vertex.x - w.vertex.x, u.vertex.y - w.vertex.y) * 1.1
							if nd < new_distance
								new_distance = [nd, old_distance].max
							end
						end
						new_distance += AVERAGE_VIA_SIZE if cur_rgt != prev_rgt # wiggly line TODO fix vertex size
						if q.inc?(w_v_rgt, new_distance)
							outer_lane[w_v_rgt] = false # we took the inner path
							parents[w_v_rgt] = min # record the path for backtracking
						end
					}
				end
				if can_out && !distances.include?(w_v_xrgt) # try outer path
					cur_rgt = !cur_rgt
					new_distance = outer_distance
					not_blocked = catch(:blocked){
						lcuts = v.vertex.neighbors - (lcuts || new_bor_list(u, w, v)) - [u.vertex, w.vertex]


						lcap = lcuts.inject(MAX_PCB_BOARD_DIA){|min, el|
							hhh = @newcuts[v.vertex, el]
							[hhh.cap - hhh.flow, min].min
						}
						throw :blocked if lcap < Trace_Width
						hhh = @newcuts[v.vertex, u.vertex]
						throw :blocked if (u != start_node) && (cur_rgt != prev_rgt) && hhh.cap - hhh.flow < Trace_Width



						#lcap = lcuts.inject(MAX_PCB_BOARD_DIA){|min, el| [@newcuts[v.vertex, el].cap, min].min}
						#throw :blocked if lcap < MinCutSize
						#throw :blocked if (u != start_node) && (cur_rgt != prev_rgt) && @newcuts[v.vertex, u.vertex].cap < MinCutSize
						throw :blocked if uv_blocked[(cur_rgt ? 1 : 0)]
						# now test special case when we touch a terminal with incident nets
						v.vertex.incident_nets.map{|el| el.nstep || el.pstep}.each{|el|
							throw :blocked if lcuts.include?(el.vertex)
							throw :blocked if !(el.nstep && el.pstep) && (cur_rgt != prev_rgt) && (el.vertex == u.vertex)
						}
						new_distance += AVERAGE_VIA_SIZE if cur_rgt != prev_rgt # TODO fix vertex size
						if q.inc?(w_v_xrgt, new_distance)
							outer_lane[w_v_xrgt] = true
							parents[w_v_xrgt] = min
						end
					}
				end
			end
		end
		path = Array.new
		p = min
		while p
			if n = parents[p]
				fail unless n[0] == p[1]
				n[0].outer = outer_lane[p]
				n[0].lr_turn = p[2] == outer_lane[p]
			end
			path << p[0]
			p = n
		end
		cid = path.last.vertex.cid
		if cid != -1 # ignore steps along edges of start cluster
			while path[-2].vertex.cid == cid
				path.pop
			end
		end

		path.each_cons(3){|w, v, u| # inverted direction
			#rgt = v.outer == v.lr_turn
			lcuts = new_bor_list(u, w, v) # neighbours in the inner angle/lane
			if v.outer 
				lcuts = v.vertex.neighbors - lcuts - [u.vertex, w.vertex]
			end
			lcuts.each{|el| @newcuts[v.vertex, el].flow += Trace_Width}
			if (v.outer == v.lr_turn) != (u.outer == u.lr_turn)
				@newcuts[v.vertex, u.vertex].flow += Trace_Width
			end 
		}
		path.first.outer = path.first.lr_turn = path.last.outer = path.last.lr_turn = nil
		return path
	end

	#          /neighbor
	#     a   /
	# <-------\
	#       /  \b
	#      /    \
	#return these
	def split_neighbor_list(a, b, n)
		fail unless a.is_a? Region
		fail unless b.is_a? Region
		nx = n.vertex.x
		ny = n.vertex.y
		aax = nx - a.vertex.x - a.ox
		aay = ny - a.vertex.y - a.oy
		bbx = b.vertex.x - nx + b.ox
		bby = b.vertex.y - ny + b.ox
		n.neighbors.select{|el|
			if el == a || el == b
				false
			else
				ex = el.vertex.x + el.ox
				ey = el.vertex.y + el.oy
				if aax * bby > aay * bbx
					aax * (ey - ny) > aay * (ex - nx) && bbx * (ey - ny) > bby * (ex - nx)
				else
					aax * (ey - ny) > aay * (ex - nx) || bbx * (ey - ny) > bby * (ex - nx)
				end
			end
		}
	end

#	def cross(o, a, b)
#		(a.x - o.x) * (b.y - o.y) - (a.y - o.y) * (b.x - o.x)
#	end

	def atan2_tangents(a, b, id)
		last_step, cur_step = a.net(id), b.net(id)
		t1 = get_tangents(a.x, a.y, last_step.radius, last_step.rgt, b.x, b.y, cur_step.radius, cur_step.rgt)
		#if a.id == 48 || b.id == 48
		#gen_line(t1[0], t1[1], t1[2], t1[3], 9)
		#end
		#t2 = get_tangents(b.x, b.y, cur_step.radius, cur_step.rgt, c.x, c.y, nxt_step.radius, nxt_step.rgt)
		Math.atan2(t1[3] - t1[1], t1[2] - t1[0])
		#Math.atan2(t1[1] - t1[3], t1[0] - t1[2])
		#((t1[2] - t1[0]) * (t2[3] - t2[1]) - (t1[3] - t1[1]) * (t2[2] - t2[0])) > 0
	end

#     a
#    /
#   /   select these neighbors of n
#  /    in inner angle < PI
# n_______b
	def new_bor_list(a, b, n)
		a, b, n = a.vertex, b.vertex, n.vertex
		a, b = b, a if RM.boolean_really_smart_cross_product_2d_with_offset(b, a, n)
		#a, b = b, a if !RM.boolean_really_smart_cross_product_2d_with_offset(a, b, n)
		n.neighbors.select{|el|
			if el == a || el == b
				false
			else
				RM.boolean_really_smart_cross_product_2d_with_offset(a, el, n) &&
				RM.boolean_really_smart_cross_product_2d_with_offset(el, b, n)
			end
		}
	end

# r1, r2 are the just splitted regions due to the newly attached net
# path r1 <-> x should be blocked, because it would cross the incident net
#    | /x
# -- r-----
#  r1||r2
	def block_paths(r1, r2, r)
		n = r.neighbors
		fail unless n.include?(r1)
		fail unless n.include?(r2)
		fail if n.include?(r)
		n.each{|el| el.a = Math.atan2(el.vertex.y - r.vertex.y, el.vertex.x - r.vertex.x)}
		z = r1.a
		fail unless z
		n.each{|el| if el.a < z then el.a += 2 * Math::PI; end}
		r2.a = r1.a + 2 * Math::PI
		n.sort_by{|el| el.a}.combination(2).each{|a, b|
			if a.vertex == b.vertex || RM::boolean_really_smart_cross_product_2d_with_offset(b, a, r)
				r.restricted_pairs << [a, b]
				r.restricted_pairs.delete([b, a])
			end
		}
	end

# Explanation of the offset ox, oy used below
# -------------------------------------------
# We have a splitted region graph shown below,
# a result of a routed trace from X to Z.
# Region Y was splitted in regions y1 and y2.
#                         /B 
#     /---------------- y1
# O--X------------------y2 \
#                       / \ \
#        				  		 /	 \ \
#                     A     Z
#
# One more trace along X--y1 will introduce one more region split
# along this path -- we split into neighbors to the left and to the right
# of the path. For neighbors A and B this is no problem, one is on the left,
# and one on the right of the path. 
# Problem: The vertex of regions y1 and y2 is the same with identical cooordinates.
# When next trace is from O over X, y1 to B, y2 should be on the right side of
# the path. We add to each fresh split region a small delta offset perpendicular
# to the path, this allows a simple decision which is on the right or left side.
# This may not work when we have a 2 PI full turn -- currently in that case offset
# is zero. We may fix that if we use a more complicated offset calculation, but indeed
# for the full turn we do not need an offset. Indeed we need the offset only for the
# first and last splitted region of a path, not for inner regions. And a full turn
# should not occur for the first or last split.
#
# Splitting the path
# ------------------
# p, c, n -- previous, current, next region; c is splitted into r1 and r2
#
#  p       p        p
#  |      / \      / \
# -c-   -r1 r2-  -r1 r2-
#  |      \ /      |  |
#  n       n     -r1'r2'
#  |       |       \ /   
#                   m
#
	def route(net_id)
		fail if net_id >= @netlist.length
		net_desc = @netlist[net_id]
		from, to = net_desc.name1, net_desc.name2
		fail unless from && to # class is string or maybe integer?
		fail if from == to
		fail unless start_node = @regions.find{|r| r.incident && r.vertex.name == from}
		#fail unless @regions.index{|r| r.incident && r.vertex.name == from} == @regions.rindex{|r| r.incident && r.vertex.name == from}
		unless path = dijkstra(start_node, to)
			puts 'dijkstra failed!'
			return
		end
		fail if path.uniq!
		fail if (bas = path.length - 1) < 1
		first = path[-1]
		last = path[0]
		fail if first == last
		r1 = r2 = nil
		prv = first
		@pic.move_to(prv.vertex.x, prv.vertex.y)
		while (bas -= 1) > 0 do
			cur = path[bas]
			nxt = path[bas - 1]
			fail unless prv && cur && nxt 
			ne = split_neighbor_list(prv, nxt, cur)
			ne_comp = ((cur.neighbors - ne) - [prv, nxt])
			ne << nxt
			ne_comp << nxt
			if r1
				ne.delete(r1)
				ne.delete(r2)
				ne_comp.delete(r1)
				ne_comp.delete(r2)
				ne << r1
				ne_comp << r2
			else
				ne << prv
				ne_comp << prv
			end
			@regions.delete(cur)
			r1 = Region.new(cur.vertex)
			r2 = Region.new(cur.vertex)
			r1.restricted_pairs = cur.restricted_pairs.dup
			r2.restricted_pairs = cur.restricted_pairs.dup
			cur.neighbors.each{|el|
				sub = Set.new
				el.restricted_pairs.delete_if{|a, b|
					if a == cur
						sub << [r1, b] << [r2, b]
					elsif b == cur
						sub << [a, r1] << [a, r2]
					end
					(a == cur) || (b == cur)
				}
				el.restricted_pairs |= sub
			}
			r1.incident =	r2.incident = cur.incident
			# give r1 and r2 an offset vector perpendicular to the path to allow a distinction
			if nxt.vertex != prv.vertex # h == 0
				r1.g = r2.g = cur.g * 0.5 # magnitude of additional offset decreases for each split by factor 0.5 
				dx1 = nxt.vertex.x - cur.vertex.x
				dy1 = nxt.vertex.y - cur.vertex.y
				h = Math::hypot(dx1, dy1)
				dx1 /= h
				dy1 /= h
				dx2 = cur.vertex.x - prv.vertex.x
				dy2 = cur.vertex.y - prv.vertex.y
				h = Math::hypot(dx2, dy2)
				dx2 /= h
				dy2 /= h
				dy = (dx1 + dx2)
				dx =  -(dy1 + dy2)
				h = Math::hypot(dx, dy) / cur.g # zero for full 2 PI turn
				dx /= h 
				dy /= h
				r1.ox = cur.ox + dx
				r1.oy = cur.oy + dy
				r2.ox = cur.ox - dx
				r2.oy = cur.oy - dy
			end
			@regions << r1 << r2
			cur.neighbors.each{|el| el.neighbors.delete(cur)}
			ne.each{|el|
				el.neighbors << r1
				r1.neighbors << el
			}
			ne_comp.each{|el|
				el.neighbors << r2
				r2.neighbors << el
			}
			r1.restricted_pairs.keep_if{|a, b| r1.neighbors.include?(a) && r1.neighbors.include?(b)}
			r2.restricted_pairs.keep_if{|a, b| r2.neighbors.include?(a) && r2.neighbors.include?(b)}
			block_paths(r2, r1, nxt) if nxt == last
			block_paths(r1, r2, prv) if prv == first
			if cur.lr_turn != cur.outer
				r1.incident = false
			else
				r2.incident = false
			end
			@pic.line_to(cur.vertex.x, cur.vertex.y)
      prv = cur
		end
		@pic.line_to(last.vertex.x, last.vertex.y)
		@pic.stroke
		#path.each{|el| el.vertex.outer = el.outer; el.vertex.lr_turn = el.lr_turn}
		#path.map!{|el| el.vertex}
		pstep = nil
		path.each_with_index{|cur, i|  
			nxt = (i == path.length - 1 ? nil : path[i + 1])
			prv = (i == 0 ? nil : path[i - 1])
			nv = (nxt ? nxt.vertex : nil)
			pv = (prv ? prv.vertex : nil)
			cv = cur.vertex
			step = Step.new(pv, nv, @path_ID)
			step.outer = cur.outer
			step.lr_turn = !cur.lr_turn
			step.net_desc = net_desc
			step.vertex = cv
			step.pstep = pstep
			pstep = step
			if prv and nxt
				cv.update(step) # TODO: if one vertex includes his neighbor vertex, then skip that one!
				step.rgt = step.outer != cur.lr_turn
				step.xt = !step.outer
				cv.attached_nets << step
			else
				step.rgt = false
				cv.incident_nets << step
			end
		}
		@path_ID += 1
		while p = pstep.pstep
			p.nstep = pstep
			pstep = p
		end
	end

	# http://en.wikipedia.org/wiki/Tangent_lines_to_circles
	# http://www.ambrsoft.com/TrigoCalc/Circles2/Circles2Tangent_.htm
	# https://en.wikibooks.org/wiki/Algorithm_Implementation/Geometry/Tangents_between_two_circles
	# UGLY: what when tangent does not exist?
	def get_tangents(x1, y1, r1, l1, x2, y2, r2, l2)
	puts 'haeh'
	puts x1, y1, r1, l1, x2, y2, r2, l2
		#d_sq = (x1 - x2) ** 2 + (y1 - y2) ** 2
		#if d_sq <=  (r1 - r2) ** 2
		#	fail 'get_tangents: circles fully overlap!'
		#end
		d = Math.hypot(x1 - x2, y1 - y2)
		vx = (x2 - x1) / d
		vy = (y2 - y1) / d
		r2 *= (l1 == l2  ? 1 : -1)
		c = (r1 - r2) / d
		h = 1 - c ** 2
		if h < 0
puts r1, r2, c, d, h
#fail
		end
		#fail if h < 0
		if h >= 0
		h = 0 if h < 0 # wrong -- only for inspection
		h = Math.sqrt(h) * (l1  ? -1 : 1)
		else
h=0
		end
		puts 'hhhhhhhhh', h
		nx = vx * c - h * vy
		puts vx , c , h , vy
		ny = vy * c + h * vx
		puts x1 , r1 , nx, y1 , r1 , ny, x2 , r2 , nx,  y2 , r2 , ny
		[x1 + r1 * nx, y1 + r1 * ny, x2 + r2 * nx,  y2 + r2 * ny]

	end

	def smart_replace(step, list)
		if step.prev == step.next # can occur due to nubly()
			#fail # can this really occur?
			fail unless list.empty? # should be empty?
			step.pstep.nstep = step.nstep.nstep
			step.pstep.next = step.nstep.next
			step.nstep.nstep.pstep = step.pstep
			step.nstep.nstep.prev = step.prev
			step.next.new_delete_net(step.nstep)
		elsif list.empty?
			ps = step.pstep
			ns = step.nstep
 			ps.next = step.next
 			ns.prev = step.prev
			ps.nstep = ns
			ns.pstep = ps
		else
		  pstep = step.pstep
			pv = step.prev
			list.each{|v|
				n = Step.new(pv, nil, step.id)
				n.net_desc = step.net_desc
				n.vertex = v
				n.pstep = pstep
				pstep.nstep = n
				pstep.next = v 
				pstep = n
				pv = v
				n.rgt = !step.rgt
				n.xt = true # TODO: check
				n.outer = true
				v.update(n)
				v.attached_nets << n
			}
			pstep.next = step.next
			pstep.nstep = step.nstep
			pstep.nstep.prev = pv
			pstep.nstep.pstep = pstep
		end
		step.vertex.new_delete_net(step)
	end

	#\   |   /
	# \  |  /
	#  \ | /  3 attached concave nets not overlapping
	#    O ------------
	#     \ -----
	#      \
	#       \
	#        \
	# sort attached nets and calculate its radii.
	# this should work for concave (before attachment operator is applied) and convex final nets
	# regard groups by angle: overlapping angles needs different radii for arcs,
	# non overlapping attached arcs may have the same radius.
	# generally one terminal should have at least two final convex groups -- but a terminal with
	# a very big radius may have more than 2 attached groups.
	# Indeed nets never cross, so overlapping is full including
	# currently we collect all attached nets of a terminal in one single array called
	# attached_nets -- we may consider using a separate array for each group...
	# maybe we should regard minimal gap as overlapping? Introduce an eps?
	# UGLY: Unused
	def currently_unused_smart_prepare_steps
		@vertices.each{|vert|
			next if vert.attached_nets.empty?
			vert.attached_nets.each{|s|
				s.g = -1
				#a = Math.atan2(s.next.y - s.vertex.y, s.next.x - s.vertex.x) # basic angle
				#b = Math.atan2(s.prev.y - s.vertex.y, s.prev.x - s.vertex.x)
				a = atan2_tangents(s.vertex, s.next, s.id) # -PI <= angle <= PI
				b = atan2_tangents(s.prev, s.vertex, s.id) # wrong direction, we have to flip
				b < 0 ? b += Math::PI : b -= Math::PI
				d = (a - b).abs
				dir = (a + b) * 0.5
				if  d > Math::PI # select inner angle < PI -- and flip direction
					d = 2 * Math::PI - d
					dir < 0 ? dir += Math::PI : dir -= Math::PI
				end
				s.d = d * 0.5
				s.dir = dir
			}
			nets = vert.attached_nets.sort_by{|s| s.dir}
			i = nets.length
			last = nets[0]
			last.dir += 2 * Math::PI
			last.g = group = 0
			while (i -= 1) > 0 do # backward while in group 0, start with largest angle
				nxt = nets[i]
				break if nxt.dir + nxt.d < last.dir - last.d
				nxt.g = 0
				last = nxt
			end
			i = 0
			last = nets[0]
			last.dir -= 2 * Math::PI # undo above fix
			while (i += 1) < nets.length do # forward
				nxt = nets[i]
				break if nxt.g == 0
				if nxt.dir - nxt.d > last.dir + last.d
					group += 1
				end
				nxt.g = group
				last = nxt
			end
			group = 0
			loop do
				vert.reset_initial_size
				group_found = false
				vert.attached_nets.each{|step|
					if step.g == group
						group_found = true
						net = step.net_desc
						trace_sep = [vert.separation, net.separation].max
						vert.radius += trace_sep + net.thickness
						step.radius = vert.radius - net.thickness * 0.5
						vert.separation = net.separation
					end
				}
				break unless group_found
				group += 1
			end
		}
	end

	# sort attached nets and calculate its radii
	def prepare_steps
		@vertices.each{|vert|
			next if vert.attached_nets.empty?
			vert.reset_initial_size
			[true, false].each{|b|
				vert.attached_nets.each{|step|
					next if step.xt == b
					net = step.net_desc
					trace_sep = [vert.separation, net.separation].max
					vert.radius += trace_sep + net.thickness
					step.radius = vert.radius - net.thickness * 0.5
					vert.separation = net.separation
				}
			}
		}
	end

	def sort_attached_nets
		@vertices.each{|vert| vert.sort_attached_nets}
	end

	def convex_kkk(prev_step, step, nxt_step)
		pv, cv, nv = step.prev, step.vertex, step.next
		x1, y1, x2, y2	= get_tangents(pv.x, pv.y, prev_step.radius, prev_step.rgt, cv.x, cv.y, step.radius, step.rgt)
		x3, y3, x4, y4	= get_tangents(cv.x, cv.y, step.radius, step.rgt, nv.x, nv.y, nxt_step.radius, nxt_step.rgt)
		x2, y2, x3, y3 = line_line_intersection(x1, y1, x2, y2, x3, y3, x4, y4) # get crossing point and ua, ub
				if (x2 != nil) && ((x3 > 0 && x3 < 1) || (y3 > 0 && y3 < 1))
			return x2, y2
		else
			return nil
		end
	end

  def nubly(collapse = false)
	  replaced = true
		rep_c = 0
		while replaced do
			replaced = false
			rep_c += 1
			@vertices.each{|cv|
				cv.attached_nets.reverse_each{|step|
					prev_step, nxt_step = step.pstep, step.nstep

					pv, nv = step.prev, step.next
					d = Math::hypot(cv.x - pv.x, cv.y - pv.y) - (prev_step.radius - step.radius).abs * 1.02
					if d < 0
						if step.radius < prev_step.radius
							step.radius -= d
							replaced = true
						end
						next
					end
					d = Math::hypot(cv.x - nv.x, cv.y - nv.y) - (nxt_step.radius - step.radius).abs * 1.02
					if d < 0
						if step.radius < nxt_step.radius
							step.radius -= d
							replaced = true
						end
						next
					end

					hx, hy = convex_kkk(prev_step, step, nxt_step)
					step.xt = hx != nil
					if collapse && step.xt
						pv, nv = step.prev, step.next
						hv0 = Vertex.new(hx, hy)

						#fail if pv == nv
						replaced = true
						pvx = pv.x
						pvy = pv.y
						nvx = nv.x
						nvy = nv.y
						if pp = prev_step.pstep
							hx, hy = convex_kkk(pp, prev_step, step)
						end
						if pp && hx
							ppv = Vertex.new(hx, hy)
						else
							ppv = pv
						end
						if nn = nxt_step.nstep
							hx, hy = convex_kkk(step, nxt_step, nn)
						end
						if nn && hx
							nnv = Vertex.new(hx, hy)
						else
							nnv = nv
						end
						hx = nvx - pvx
						hy = nvy - pvy
						if step.rgt
							vec_x, vec_y = hy, -hx
						else
							vec_x, vec_y = -hy, hx
						end
						hv3 = Vertex.new(pvx + hx * 0.5 + vec_x, pvy + hy * 0.5 + vec_y)
						hx *= 2
						hy *= 2
						vec_x *= 2
						vec_y *= 2
						hv4 = Vertex.new(pvx - hx + vec_x, pvy - hy + vec_y)
						hv5 = Vertex.new(nvx + hx + vec_x, nvy + hy + vec_y)
						rep = vertices_in_polygon([ppv, hv0, nnv, hv3], @vertices) - [pv, nv, ppv, cv, nnv, hv3]
						unless rep.empty?
							net = step.net_desc
							rep.each{|v|
								v.trgt = !step.rgt
								v.tradius = v.radius + [net.separation, v.separation].max + net.thickness * 0.5
							}
							pv.trgt = step.pstep.rgt
							pv.tradius = step.pstep.radius
							nv.trgt = step.nstep.rgt
							nv.tradius = step.nstep.radius
							rep = new_convex_vertices(rep, pv, nv, hv4, hv5)
						end
						smart_replace(step, rep)
					end
				}
			}
		end
	end

	def draw_routes

		@file = File.open("layout_data.pcb", "w")	
		#@file.write()
 	@file.write("FileVersion[20070407]\n")
 	@file.write("PCB[\"\" 350000 330000]\n")
 	@file.write("Grid[3900.0 1800 100 1]\n")
	gen_vias
 	@file.write("Layer(1 \"solder\")\n(\n")



		set_color(0, 0, 0, 1)
		@vertices.each{|vert|
			vert.incident_nets.each{|n|
				if n.next
					@pic.new_sub_path
					@pic.arc(vert.x, vert.y, 100, 0, 2 * Math::PI)
					@pic.stroke
				end
				last = vert
				lastx = lasty = nil
				lr = 0
				to = n.next
				to_net = n.nstep
				while to do
					last_net = to_net.pstep
					last = last_net.vertex
					radius = to_net.radius
					if last.x == to.x && last.y == to.y
					last.vis_flag = 1
					else
					t = get_tangents(last.x, last.y, lr, last_net.rgt, to.x, to.y, radius, to_net.rgt)
					gen_line(*t, Trace_Width)
					if lr > 0
					  start_angle = Math.atan2(lasty - last.y, lastx - last.x)
						end_angle = Math.atan2(t[1] - last.y, t[0] - last.x)
						start_angle, end_angle = end_angle, start_angle unless last_net.rgt
						puts start_angle, end_angle
#@pic.set_source_rgba(1, 0, 0, 1) if last_net.outer# == last_net.lr_turn

#fail unless (last_net.outer == !last_net.lr_turn) == last_net.rgt

						gen_arc(last.x, last.y, lr, start_angle, end_angle, Trace_Width)
						@pic.set_source_rgba(0, 0, 0, 1) 
					end
					end
					lr = radius
					last = to
					to_net = to_net.nstep
					if to_net
					to = to_net.vertex
					else
						to = nil
					end
					lastx = t[2]
					lasty = t[3]
				end
			}
		}
		@file.write(")\n")
		file.close unless file == nil
	end

	def set_line_width(w)
		@pic.set_line_width(w)
	end
	def set_color(r, g, b, a)
		@pic.set_source_rgba(r, g, b, a)
	end
	def save_picture
		@image.write_to_png(@filename)
	end
end
end

if __FILE__ == $0



rs = RM::init_seed
#r = RBR::Router.new(0, 0, Board_Size, Board_Size)
r = RBR::Router.new(0, 0, PCB_Size, PCB_Size)

r.generate_test_vertices
r.finish_init(true)
r.filename = 'pic' + rs.to_s.rjust(3, "0") + '.png'
r.draw_vertices

col = ([1, 0, 0].permutation(3).to_a + [1, 1, 0].permutation(3).to_a).uniq - [[1, 0, 0]]
r.set_color(1, 0, 0, 0.7)
r.set_line_width(1200)
#(5..9).each{|i|
###(4..8).each{|i|
[0,1,4,5,6,7,8].each{|i|
#next if i == 7
#(2..2).each{|i|
r.set_color(*col[i % 5], 0.4)
r.route(i)
}
r.sort_attached_nets

r.prepare_steps
r.nubly
r.prepare_steps

r.sort_attached_nets
#r.nobly
r.prepare_steps


r.nubly
r.prepare_steps

r.sort_attached_nets
#r.nobly
r.prepare_steps






r.nubly(true)
r.sort_attached_nets
#r.nobly
r.prepare_steps









#r.nubly


#r.sort_attached_nets

#r.prepare_steps

#r.nubly


#r.sort_attached_nets

#r.prepare_steps



#r.nobly
#r.sort_attached_nets

#r.prepare_steps
#r.nobly


#r.nubly



r.draw_routes
r.flag_vertices
r.save_picture
puts $glob


end
