#!/usr/bin/ruby -w
# pet.rb -- plain electronics tool (a tiny geda/gschem clone written from scratch in Ruby) 
# This is an early draft or proof of concept -- project name may change
# We try to support recent gschem file format -- later we may export an (extended) PCB netlist
# This file will be supported by a plain Ruby GTK/Cairo schematics editor (name may be peted.rb)
# Copyright: S. Salewski, mail@ssalewski.de
# License: GPL
#
module Pet
#
require 'gtk3'
require 'cairo' # PNG, PDF, SVG export
require 'pango' # font rendering
require_relative 'pet_conf'
require_relative 'pet_bbox'
require_relative 'pet_def'
Version = '0.01 (11-AUG-2011)'
#
Docu = <<HERE
pet.rb -- a plain electronics tool inspired by the gEDA suite
Version: #{Version}
Author: S. Salewski
License: GPL

usage: ...

HERE

PROGRAM_VERSION = 'xxx'
FILEFORMAT_VERSION = 'yyy'
EGRID = 100 # pins and nets should start/end on EGRID only
MIN_STRUCTURE_SIZE = 5 # we try to prevent pollution of schematics with tiny objects

ArcChar     = 'A'
BoxChar     = 'B'
BusChar     = 'U'
CircChar    = 'V'
LineChar    = 'L'
NetSegChar  = 'N'
PathChar    = 'H'
PicChar     = 'G'
PinChar     = 'P'
SymChar     = 'C'
TextChar    = 'T'
VersionChar = 'v'

AttrPat = '(.*)=(.*)'

SOLPAT   = '^'
EOLPAT   = ' *$'
INTPAT   = ' (-?\d+)'
FLOATPAT = ' (\d+(\.\d+)?([eE][-+]?\d+)?)'

ArcPar = %w[Arc type x y radius startangle sweepangle color linewidth capstyle dashstyle dashlength dashspace]
ArcPat = SOLPAT + '(' + ArcChar + ')' + INTPAT * 11 + EOLPAT

BoxPar = %w[Box type x y width height color linewidth capstyle dashstyle dashlength dashspace filltype fillwidth angle1 pitch1 angle2 pitch2]
BoxPat = SOLPAT + '(' + BoxChar + ')' + INTPAT * 16 + EOLPAT

BusPar = %w[Bus type x1 y1 x2 y2 color ripperdir]
BusPat = SOLPAT + '(' + BusChar + ')' + INTPAT * 6 + EOLPAT

CircPar = %w[Circ type x y radius color linewidth capstyle dashstyle dashlength dashspace filltype fillwidth angle1 pitch1 angle2 pitch2]
CircPat = SOLPAT + '(' + CircChar + ')' + INTPAT * 15 + EOLPAT

SymPar = %w[Symbol type x y selectable angle mirror basename]
EmbeddedSymPat = SOLPAT + '(' + SymChar + ')' + INTPAT * 5 + ' EMBEDDED' + '(.*\.sym)' + EOLPAT
ExternSymPat   = SOLPAT + '(' + SymChar + ')' + INTPAT * 5 + ' '         + '(.*\.sym)' + EOLPAT

LinePar = %w[Line type x1 y1 x2 y2 color linewidth capstyle dashstyle dashlength dashspace]
LinePat = SOLPAT + '(' + LineChar + ')' + INTPAT * 10 + EOLPAT

PathPar = %w[Path type color linewidth capstyle dashstyle dashlength dashspace filltype fillwidth angle1 pitch1 angle2 pitch2 numlines]
PathPat = SOLPAT + '(' + PathChar + ')' + INTPAT * 13 + EOLPAT

PicPar = %w[Pic type x y width height angle ratio mirrored embedded]
PicPat = SOLPAT + '(' + PicChar + ')' + INTPAT * 5 + FLOATPAT + INTPAT * 2 + EOLPAT

PinPar = %w[Pin type x1 y1 x2 y2 color pintype whichend]
PinPat = SOLPAT + '(' + PinChar + ')' + INTPAT * 7 + EOLPAT

TextPar = %w[Text type x y color size visibility show_name_value angle alignment num_lines]
TextPat = SOLPAT + '(' + TextChar + ')' + INTPAT * 9 + EOLPAT

NetSegPar = %w[NetSeg type x1 y1 x2 y2 color]
NetSegPat = SOLPAT + '(' + NetSegChar + ')' + INTPAT * 5 + EOLPAT

VersionPar = %w[Version type version fileformat_version]
VersionPat = SOLPAT + '(' + VersionChar + ')' + INTPAT * 2 + EOLPAT

=begin
# plain source code generation -- save some typing effort
def self.docgen(c, p)
  print 'Object: ', p[0], "\n", c
  p[2..-1].each{|x| print ' ', x}
  print "\n", c, ": Type is ", p[0], "\n"
  p[2..-1].each{|x| print x, ": \n"}
  puts
end

docgen(ArcChar, ArcPar)
docgen(LineChar, LinePar)
docgen(BoxChar, BoxPar)
docgen(BusChar, BusPar)
docgen(CircChar, CircPar)
docgen(NetSegChar, NetSegPar)
docgen(PathChar, PathPar)
docgen(PicChar, PicPar)
docgen(PinChar, PinPar)
docgen(SymChar, SymPar)
docgen(TextChar, TextPar)
docgen(VersionChar, VersionPar)

Process.exit
begin
def self.ppar(a)
  a.each_with_index{|x,i| print "  el.#{x} = match[#{i}]\n"}
  print '  attr_accessor'; a.each{|x| print ' :', x, ','}; puts
  print '  pline'; a.each{|x| print ' @', x, ','}; puts
  puts
end

ppar(ArcPar)
ppar(BoxPar)
ppar(BusPar)
ppar(CircPar)
ppar(LinePar)
ppar(NetSegPar)
ppar(PathPar)
ppar(PicPar)
ppar(PinPar)
ppar(SymPar)
ppar(TextPar)
ppar(VersionPar)
Process.exit
=end

module PES # Process_Event_State
  Hoovering = 0
  Hit       = 1
  Dragging  = 2
  Moved     = 3
end

module PEM # Process_Event_Message
  Hit_Select    = 0
  Drag_Select   = 1
  Hoover_Select = 2
#  First_Delta_Move = 3
  Delta_Move    = 4
  KEY_Delete    = 5
end


Default_Line_Width_Scale = 3 # compare to EGRID==100, PIN-Length==300

#      (c)
#     /
#    /     (p)
#   /
# (b)
# see http://www.geometrictools.com/
#
def self.distance_line_segment_point_squared(bx, by, cx, cy, px, py)
  mx = cx - bx
  my = cy - by
  hx = px - bx
  hy = py - by
  t0 = (mx * hx + my * hy).fdiv(mx ** 2 + my ** 2)
  if t0 <= 0
  elsif t0 < 1
    hx -= t0 * mx
    hy -= t0 * my
  else
    hx -= mx
    hy -= my
  end
  return hx ** 2 + hy ** 2
end

def self.rtate(x, y, ox, oy, angle)
  if angle == 0 then return x, y end
  x -= ox
  y -= oy
  if angle == 90
    x, y = -y, x
  elsif angle == 180
    x, y = -x, -y
  elsif angle == 270
    x, y = y, -x
  else
    angle *= (Math::PI / 180.0)
    sin, cos = Math::sin(angle), Math::cos(angle)
    x, y = x * cos - y * sin, x * sin + y * cos 
  end
  return x + ox, y + oy
end

class Pet_Object_List < Array
  attr_accessor :selected, :hit_selected, :select_one, :hoover #, :hits
  def initialize
    super
    @hoover = false
    @selected = 0
    @hit_selected = 0
    @select_one = 0
    @hits = 0
  end

#  def hoovering?()
#    @hoover
#  end


  def selected_element()
    self.each{|el|
      if (el.state == State::Selected) then return el end
    }
    return nil
  end


  def process_event(boxlist, event, x0, y0, x, y, msg)


  if msg == nil # rectangle


    el = self[-1]
    if el and el.absorbing
      @hoover = true
      el = el.absorb(boxlist, event, x0, y0, x, y, msg)
      if el then self << el end
      return true
    else
      if (event.event_type == Gdk::Event::BUTTON_PRESS)
        self << Box.start(x, y)
        return true
      end

    end


  else



el = self[-1]
if el and el.absorbing
      @hoover = true
el = el.absorb(boxlist, event, x0, y0, x, y, msg)
if el then self << el end
return true
end


if true and (msg == PEM::Drag_Select)


	puts 'pevent4'
	puts msg
#self << Box.start(x, y) # wrong, found 2014012
#return true
end

    if msg == PEM::Delta_Move then @fdm += 1 else @fdm = 0 end
    if msg == PEM::Hoover_Select
      @hoover = false
      @select_one = 0
      self.each{|el|
        if el.bbox and (el.selectable != 0) # hidden text has no valid bbox
          el.old_hoover = el.hoover
          el.hoover = el.bbox.include_point?(x, y)
          if el.hoover
            @hoover = true
            if el.core_box
              el.hoover = el.core_box.include_point?(x, y)
            end
          end
        else # should be obsolete
          el.old_hoover = false
          el.hoover = false
          el.box_size = 0
        end
      }
    elsif (msg == PEM::Hit_Select) or (@fdm == 1)
      if (msg == PEM::Hit_Select) and (@hits > 1) and (@select_one > 0)
        @hit_selected = 1
        min = 1E6.to_i
        pos = 0
        self.each_with_index{|el, i|
          el.hoover = false
          if (el.box_size > 0) and (el.box_size < min)
            min = el.box_size
            pos = i
          end
        }
        self[pos].hoover = true
        self[pos].box_size += 1E4.to_i
      else
        @hits = 0
        @hit_selected = 0
        self.each{|el|
          if el.bbox and (el.selectable != 0) # hidden text has no valid bbox
            if el.hoover
              @hits += 1
              if (el.state == State::Selected) then @hit_selected += 1 end
              el.box_size = el.bbox.hit_size()
            else
              el.box_size = 0
            end
          end
        }
        if (msg == PEM::Hit_Select) then @select_one += 1 end
      end
    end
    res = nil
    absorbed = false
    objlist = Array.new
    self.each{|el|
      res = el.process_event(boxlist, objlist, event, x0, y0, x, y, @hit_selected, msg)
      if res
        absorbed = true
        if (res.class == NetSegment)
          #objlist << res
          break
        end
      end
    }
    if res and (res.class == NetSegment)
    self << res
    end
    return absorbed
  end
  end

=begin
  def process_event(boxlist, event, x0, y0, x, y, msg)
    if msg == PEM::Hoover_Select
      @hits = 0
      @hit_selected = 0
      @hoover = false
      @select_one = 0#-1
      self.each{|el|
        if el.bbox and (el.selectable != 0) # hidden text has no valid bbox
          el.old_hoover = el.hoover
          el.hoover = el.bbox.include_point?(x, y)
          if el.hoover
            @hoover = true
            if el.core_box
              el.hoover = el.core_box.include_point?(x, y)
            end
          end
          if el.hoover
            if (el.state == State::Selected) then @hit_selected += 1 end
            #@hit_selected += 1 if (el.state == State::Selected)
            @hits += 1
            el.box_size = el.bbox.hit_size()
          else
            el.box_size = 0
          end
        else # should be obsolete
          el.old_hoover = false
          el.hoover = false
          el.box_size = 0
        end
      }
    elsif msg == PEM::Hit_Select
      if (@hits > 1) and (@select_one > 0)
        @hit_selected = 1
        min = 1E6.to_i
        pos = 0
        self.each_with_index{|el, i|
          el.hoover = false
          el.old_hoover = false
          if (el.box_size > 0) and (el.box_size < min)
            min = el.box_size
            pos = i
          end
        }
        self[pos].hoover = true
        self[pos].box_size += 1E4.to_i
      end
      @select_one += 1
    end
    absorbed = false
    objlist = Array.new
    self.each{|el| if el.process_event(boxlist, objlist, event, x0, y0, x, y, @hit_selected, msg) then absorbed = true end}
    return absorbed
  end
=end

end

class Cairo::Context
  attr_accessor :sharp_lines, :soft, :highlight, :line_width_device_min, :line_width_unscaled_user_min, :line_width_scale, :text_shadow_scale, :line_shadow_fix, :text_shadow_fix
  attr_accessor :bbox, :g_join, :g_cap, :connections, :hair_line_scale
attr_accessor :background_pattern, :background_pattern_offset_x, :background_pattern_offset_y, :device_grid_major
  alias :orig_init :initialize
  def initialize(surface)
    orig_init(surface)
@background_pattern = nil
@device_grid_major = 1
@background_pattern_offset_x, @background_pattern_offset_y = 0, 0
    @bbox = Bounding::Box.new_ghost
    @connections = Hash.new(0)
    @sharp_lines = true # defaults, overwrite for each surface if necessary
    @soft = false
    @highlight = false
    @line_shadow_fix = 0
    @text_shadow_fix = 0
    @text_shadow_scale = 1
    @line_width_device_min = 1
    @hair_line_scale = 0.2
    @line_width_unscaled_user_min = 5#1
    @line_width_scale = Default_Line_Width_Scale
    @g_join = Array.new
    @g_join[GEDA::END_CAP[:NONE]] = Cairo::LINE_JOIN_BEVEL
    @g_join[GEDA::END_CAP[:SQUARE]] = Cairo::LINE_JOIN_MITER
    @g_join[GEDA::END_CAP[:ROUND]] = Cairo::LINE_JOIN_ROUND
    @g_cap = Array.new
    @g_cap[GEDA::END_CAP[:NONE]] = Cairo::LINE_CAP_BUTT
    @g_cap[GEDA::END_CAP[:SQUARE]] = Cairo::LINE_CAP_SQUARE
    @g_cap[GEDA::END_CAP[:ROUND]] = Cairo::LINE_CAP_ROUND
  end

  def geda_set_dash(dashstyle, dashlength, dashspace)
    if dashstyle == GEDA::LINE_TYPE[:SOLID]
      set_dash([], 0)
    elsif dashstyle == GEDA::LINE_TYPE[:DOTTED]
      set_line_cap(Cairo::LINE_CAP_ROUND)
      set_dash([0, dashspace], 0)
    elsif dashstyle == GEDA::LINE_TYPE[:DASHED]
      set_dash([dashlength, dashspace], dashlength * 0.5)
    elsif dashstyle == GEDA::LINE_TYPE[:CENTER]
      set_line_cap(Cairo::LINE_CAP_ROUND)
      set_dash([dashlength, dashspace, 0, dashspace], dashlength * 0.5)
    elsif dashstyle == GEDA::LINE_TYPE[:PHANTOM]
      set_line_cap(Cairo::LINE_CAP_ROUND)
      set_dash([dashlength, dashspace, 0, dashspace, 0, dashspace], dashlength * 0.5)
    end
  end

#  def geda_set_cap(cap)
#    if cap == GEDA::END_CAP[:NONE]
#      set_line_cap(Cairo::LINE_CAP_BUTT)
#    elsif cap == GEDA::END_CAP[:SQUARE]
#      set_line_cap(Cairo::LINE_CAP_SQUARE)
#    elsif cap == GEDA::END_CAP[:ROUND]
#      set_line_cap(Cairo::LINE_CAP_ROUND)
#    end
#  end

#  def geda_set_join(cap)
#    if cap == GEDA::END_CAP[:NONE]
#      set_line_join(Cairo::LINE_JOIN_BEVEL)
#    elsif cap == GEDA::END_CAP[:SQUARE]
#      set_line_join(Cairo::LINE_JOIN_MITER)
#    elsif cap == GEDA::END_CAP[:ROUND]
#      set_line_join(Cairo::LINE_JOIN_ROUND)
#    end
#  end

  if method_defined?(:my_method_is_undefined?)
    puts 'We are unintentionally overwriting method Cairo::Context.my_method_is_undefined?'
    Process.exit
  else
    def self.my_method_is_undefined?(sym)
      if method_defined?(sym)
        print 'We are unintentionally overwriting method Cairo::Context.', sym.to_s, "\n"
        Process.exit
      end
      return true
    end
  end

  if self.my_method_is_undefined?(:user_to_device_scale)
    def user_to_device_scale(w)
      x, y = user_to_device_distance(w, 0)
      return Math::sqrt(x**2 + y**2)
    end
  end

  if self.my_method_is_undefined?(:device_to_user_scale)
    def device_to_user_scale(w)
      x, y = device_to_user_distance(w, 0)
      return Math::sqrt(x**2 + y**2)
    end
  end

  if self.my_method_is_undefined?(:set_color)
    def set_color(r, g, b, a)
      if @highlight == true
        r = r ** 1.3 * 1.8
        g = g ** 1.3 * 1.8
        b = b ** 1.3 * 1.8
        m = [r, g, b].max
        if m > 1
          r /= m
          g /= m
          b /= m
        end
      end
      self.set_source_rgba(r, g, b, a)
    end
  end

  if self.my_method_is_undefined?(:unscaled_user_to_device_line_width)
    def unscaled_user_to_device_line_width(u)
      x = [u, @line_width_unscaled_user_min].max * @line_width_scale
      x, y = user_to_device_distance(x, 0)
      return [Math::sqrt(x**2 + y**2), @line_width_device_min].max
    end
  end

  if self.my_method_is_undefined?(:device_to_user_line_width)
    def device_to_user_line_width(d)
      w = [d, @line_width_device_min].max
      x, y = device_to_user_distance(w, 0)
      return Math::sqrt(x**2 + y**2)
    end
  end

  if self.my_method_is_undefined?(:line_pollution)
    def line_pollution(w)
      (0.5 * device_to_user_line_width(unscaled_user_to_device_line_width(w)) + 0.9 * device_to_user_line_width(unscaled_user_to_device_line_width(0)))
    end
  end

  if self.my_method_is_undefined?(:clamped_line_width)
    def clamped_line_width(w)
      device_to_user_line_width(unscaled_user_to_device_line_width(w + @line_shadow_fix))
    end
  end

  if self.my_method_is_undefined?(:clamped_set_line_width)
    def clamped_set_line_width(w)
      set_line_width(device_to_user_line_width(unscaled_user_to_device_line_width(w + @line_shadow_fix)))
    end
  end

  if self.my_method_is_undefined?(:sharp_line)
    def sharp_line(x1, y1, x2, y2, wu)
      if @sharp_lines
        eps = 0.1
        wd = unscaled_user_to_device_line_width(wu + @line_shadow_fix).round
        wu = device_to_user_line_width(wd)
        x1, y1 = user_to_device(x1, y1)
        x2, y2 = user_to_device(x2, y2)
        if (x1 - x2).abs < eps
          x1 = (x1 + x2) * 0.5
          if (wd % 2 == 0) != @soft
            x1 = x1.round
          else
            x1 = x1.floor + 0.5
          end
          x2 = x1
        elsif (y1 - y2).abs < eps
          y1 = (y1 + y2) * 0.5
          if (wd % 2 == 0) != @soft
            y1 = y1.round
          else
            y1 = y1.floor + 0.5
          end
          y2 = y1
        end
        x1, y1 = device_to_user(x1, y1)
        x2, y2 = device_to_user(x2, y2)
        set_line_width(wu)
      else
        clamped_set_line_width(wu)
      end
      move_to(x1, y1)
      line_to(x2, y2)
      stroke
    end
  end

  if self.my_method_is_undefined?(:sharp_thin_line)
    def sharp_thin_line(x1, y1, x2, y2, wu)
      if @sharp_lines
        eps = 0.1
        wu = [wu, @line_width_unscaled_user_min].max * @line_width_scale * @hair_line_scale
        w = user_to_device_scale(wu)
        wd = w.round
        wu = device_to_user_scale(wd) if w > 1
        if wd < 1 then wd = 1 end
        x1, y1 = user_to_device(x1, y1)
        x2, y2 = user_to_device(x2, y2)
        if (x1 - x2).abs < eps
          x1 = (x1 + x2) * 0.5
          if (wd % 2 == 0) != @soft
            x1 = x1.round
          else
            x1 = x1.floor + 0.5
          end
          x2 = x1
        elsif (y1 - y2).abs < eps
          y1 = (y1 + y2) * 0.5
          if (wd % 2 == 0) != @soft
            y1 = y1.round
          else
            y1 = y1.floor + 0.5
          end
          y2 = y1
        end
        x1, y1 = device_to_user(x1, y1)
        x2, y2 = device_to_user(x2, y2)
        set_line_width(wu)
      else
        clamped_set_line_width(wu)
      end
      move_to(x1, y1)
      line_to(x2, y2)
      stroke
    end
  end





  if self.my_method_is_undefined?(:faster_sharp_thin_line_h)
    def faster_sharp_thin_line_h(x1, x2, y, even)
      y = user_to_device(x1, y)[1]
      y = device_to_user(x1, even ? y.round : y.floor + 0.5)[1]
      move_to(x1, y)
      line_to(x2, y)
    end
  end

  if self.my_method_is_undefined?(:faster_sharp_thin_line_v)
    def faster_sharp_thin_line_v(y1, y2, x, even)
      x = user_to_device(x, y1)[0]
      x = device_to_user(even ? x.round : x.floor + 0.5, y1)[0]
      move_to(x, y1)
      line_to(x, y2)
    end
  end


  if self.my_method_is_undefined?(:sharp_rect)
    def sharp_rect(x, y, w, h, wu)
      aa = self.antialias
      if @sharp_lines
        set_antialias(Cairo::ANTIALIAS_NONE)
      end
      clamped_set_line_width(wu)
      rectangle(x, y, w, h)
      stroke
      set_antialias(aa)
    end
  end
end # class Cairo::Context

def self.on_egrid?(*p)
  p.each{|x| if x % EGRID != 0 then return false end}
  return true
end

def self.diagonal_line?(x1, y1, x2, y2)
  (x1 != x2) and (y1 != y2)
end

def self.line_too_short?(x1, y1, x2, y2)
  ((x1 - x2) ** 2 + (y1 - y2) ** 2) < MIN_STRUCTURE_SIZE ** 2
end


class Element
  attr_reader :bbox, :core_box, :type
  attr_accessor :state, :old_hoover, :hoover, :box_size, :selectable, :absorbing
  def initialize
    @state = State::Visible
    @hoover = false
    @old_hoover = false
    @box_size = 0
    @selectable = 1
    @core_box = nil
    @absorbing = false
  end

  def absorb(boxlist, event, x0, y0, x, y, msg); return nil end

  def init_connections(h) end

  def draw_junctions(cr, par, damage_list, draw_hoovered, draw_selected) end

  def check; return ''; end

  def mytos(*a)
    a.join(' ') + "\n"
  end

  def to_s
    mytos @type
  end

  def write(file)
    file.write to_s
  end

  def set_box() end

  def translate(x, y) end

  def rotate(x, y, angle) end

  def enlarge_bbox(b)
    if @bbox then b.join(@bbox); end
  end

  def check_bbox(x, y)
    (@selectable != 0) and @bbox.include_point?(x, y)
  end 

=begin
  # start new net, edit vertices...
  def special_LMBU_action(objlist, px, py)
  if self.instance_variable_defined?(:@components)
    if @state ==  State::Active
      @components.each{|c|
        if c.class == Pin
          if (cp = c.connect(px, py))
            @state = State::Visible
             net = NetSegment.start(*cp)
            objlist << net
            return true
          end
        end
      }
    end
  end
    return false
  end
=end

#  def special_move_action(objlist, px, py, dx, dy)
#    return false
#  end

=begin
  # ignore movement if pin is grabbed -- action may be bound to an overlapping net segment
  def special_move_action()
  if self.instance_variable_defined?(:@components)
    @components.each{|c|
      if c.class == Pin
        if c.connect(@px, @py)
          @state = State::Visible
          return true # absorbed and ignored
        end
      end
    }
    end
    return false
  end
=end

#  def hoovering?(px, py)
#    @bbox.include_point?(px, py, 100)
#  end


  # outdated...
  # Select, translate, delete elements -- modify vertices -- start new nets from pins and existing nets
  # Caution: For some operations on multiple selected elements (move, delete...) this methode will not
  # work because selected elements become deselected if @bbox.include_point?(px, py) == false -- we
  # have to check for multiple selections first and opperate on whole collection if necessary.
  # If we generate (and return) a new net segment, then the caller should stop processing remaining elements.
  # It may be good to have the ability to work on selected elements only, so that the user can
  # selective work with overlapping elements (and net nodes). Not implemented yet -- one solution is to
  # call this methode with all selected elements first, and use an optional second pass for remaning elements.


  # returns true if element is moved or state is changed
  def process_event(boxlist, objlist, event, x0, y0, x, y, hit_selected, msg)
    if ((event.event_type == Gdk::Event::Type::BUTTON_PRESS) or (event.event_type == Gdk::Event::Type::BUTTON_RELEASE)) and (event.button != 1)
      return false # (currently) button 2/3 is ignored 
    end
    #if @hoover and self.instance_variable_defined?(:@attributes) and (@attributes.length > 0)
    if self.instance_variable_defined?(:@attributes) and (@attributes.length > 0)
      unless (msg ==  PEM::Drag_Select) and (@bbox and Bounding::Box.new(x0, y0, x, y).include?(@bbox))
        if not ((hit_selected > 0) and (@state == State::Selected))
          if @attributes.process_event(boxlist, event, x0, y0, x, y, msg)
            if @core_box
              @bbox = @core_box.dup
              @attributes.each{|x| x.enlarge_bbox(@bbox) unless x == nil}
            end
            return true
          end
        end
      end  
    end
    ctrl = ((event.state & Gdk::Window::ModifierType::CONTROL_MASK) != 0)
    shift = ((event.state & Gdk::Window::ModifierType::SHIFT_MASK) != 0)
    old_state = @state
    if (msg == PEM::Hit_Select) or (msg ==  PEM::Drag_Select)
      if (msg == PEM::Hit_Select) and self.respond_to?(:special_hit_action) and (new_el = special_hit_action(x, y))
        return new_el
      end
      if msg == PEM::Hit_Select
        sel = @hoover
      else # msg ==  PEM::Drag_Select
        sel = (@bbox and Bounding::Box.new(x0, y0, x, y).include?(@bbox))
      end
      if sel
        if ctrl # toggle
          if @state == State::Visible
            @state = State::Selected
          else
            @state = State::Visible
          end
        else # add
          @state = State::Selected
        end
      elsif not (ctrl or shift)
        @state = State::Visible
      end
    elsif (msg == PEM::Delta_Move)# or (msg == PEM::First_Delta_Move)
      if @hoover and self.respond_to?(:special_move_action) and special_move_action(objlist, x0, y0, x, y)
        old = @bbox.dup()
        set_box()
        boxlist << old.join(@bbox)
        return true
      elsif ((hit_selected > 0) and (@state == State::Selected)) or ((hit_selected == 0) and @hoover)
        nb = @bbox.dup
        nb.enlarge(x, y)
        boxlist << nb
        self.translate(x, y)
        return true
      end
    end
    if (@state != old_state) or (@hoover != @old_hoover)
      boxlist << @bbox
    end
    return (@state != old_state)
  end
end

class Versio < Element
  attr_accessor :version, :fileformat_version
  def initialize
    super
    @type = VersionChar
    @version = PROGRAM_VERSION
    @fileformat_version = FILEFORMAT_VERSION
  end
  def to_s
    mytos @type, @version, @fileformat_version
  end
  def write(file)
    file.write to_s
  end
  def draw(cr, par, damage_list, draw_hoovered, draw_selected); end
  def enlarge_bbox(b); end
end



class Line < Element
  attr_accessor :x1, :y1, :x2, :y2, :color, :width, :capstyle, :dashstyle, :dashlength, :dashspace
  def initialize
    super
    @type = LineChar
    @x1, @y1, @x2, @y2, @color, @width = 0, 0, 0, 0, 0, 0 # overwrite, or check() will complain
    @capstyle = GEDA::END_CAP[:ROUND]
    @dashstyle = GEDA::LINE_TYPE[:SOLID]
    @dashlength = -1
    @dashspace = -1
  end
  def check
    if @width < 0
      return "line width should not be negative (#{@width})\n"
    elsif not GEDA::END_CAP.has_value? @capstyle
      return "unsupported cap style (#{@capstyle})\n"
    elsif not GEDA::LINE_TYPE.has_value? @dashstyle
      return "unsupported dash style (#{@dashstyle})\n"
    elsif (@dashstyle != GEDA::LINE_TYPE[:SOLID]) and (@dashspace <= 0)
      return "dashspace should be >= 0 (#{@dashspace})\n" # we should query result of cairo stroke for other problems 
    elsif not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n" 
    elsif Pet.line_too_short?(@x1, @y1, @x2, @y2)
      return "very short line, #{@x1}, #{@y1} -- #{@x2}, #{@y2}\n"
    else
      return ''
    end
  end
  def to_s
    mytos @type, @x1, @y1, @x2, @y2, @color, @width, @capstyle, @dashstyle, @dashlength, @dashspace
  end
  def write(file)
    file.write to_s
  end
  def set_box()
    @bbox = Bounding::Box.new(@x1, @y1, @x2, @y2).grow(32)
  end
  def translate(x, y)
    @x1 += x; @x2 += x; @y1 += y ;@y2 += y;
    @bbox.translate(x, y)
  end
  def rotate(x, y, angle)
    @x1, @y1 = Pet.rtate(@x1, @y1, x, y, angle)
    @x2, @y2 = Pet.rtate(@x2, @y2, x, y, angle)
    set_box
  end
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)

  if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
  if not @bbox.overlap_list?(damage_list) then return end
    cr.set_color(*par[Pet_Config::CIM[@color]])
    cr.set_line_cap(cr.g_cap[capstyle])
    cr.geda_set_dash(dashstyle, dashlength, dashspace)
    cr.sharp_line(x1, y1, x2, y2, @width)
    @bbox = Bounding::Box.new(@x1, @y1, @x2, @y2).grow(cr.line_pollution(@width))
  end
end

class NetSegment < Element
  attr_accessor :x1, :y1, :x2, :y2, :color
  def initialize(x1, y1, x2 = nil, y2 = nil)
    super()
    if x2 == nil then x2 = x1 end
    if y2 == nil then y2 = y1 end
    @type = NetSegChar
    @x1, @y1, @x2, @y2 = x1, y1, x2, y2
    @color = Pet_Config::Colorindex_geda_net
    self.set_box()
  end

  def init_connections(h)
    h[[@x1, @y1]] += 1
    h[[@x2, @y2]] += 1
  end

  def NetSegment.start(x, y)
n = NetSegment.new(x, y)
    n.absorbing = true
    return n
  end

  def connect(px, py)
    if (@x1 - px)**2 + (@y1 - py)**2 < 50**2
      return [@x1, @y1]
    elsif (@x2 - px)**2 + (@y2 - py)**2 < 50**2
      return [@x2, @y2]
    else
      return nil
    end
  end

=begin
  def special_LMBU_action(objlist, px, py)
    if @state == State::Active
      if (cp = connect(px, py))
        net = NetSegment.start(*cp)
        objlist << net
        return true
      end
    end
return false
  end

  # start new net, edit vertices...
  def special_LMBD_action(objlist, px, py)
return false
    @components.each{|c|
      if c.class == Pin
        if (cp = c.connect(px, py))
          net = NetSegment.start(*cp)
          objlist << net
          return true
        end
      end
    }
    return false
  end
=end







  def special_move_action(objlist, px, py, dx, dy)
    if (@x1 - px)**2 + (@y1 - py)**2 < 50**2
      @x1 += dx
      @y1 += dy
      return true
    elsif (@x2 - px)**2 + (@y2 - py)**2 < 50**2
      @x2 += dx
      @y2 += dy
      return true
    end
    return false
  end

  def no_process_event(boxlist, objlist, event, x0, y0, x, y, hit_selected, msg)
    if ((event.event_type == Gdk::Event::BUTTON_PRESS) or (event.event_type == Gdk::Event::BUTTON_RELEASE)) and (event.button != 1)
      return false # (currently) button 2/3 is ignored 
    end
    #if @hoover and self.instance_variable_defined?(:@attributes) and (@attributes.length > 0)
    if self.instance_variable_defined?(:@attributes) and (@attributes.length > 0)
      unless (msg ==  PEM::Drag_Select) and (@bbox and Bounding::Box.new(x0, y0, x, y).include?(@bbox))
        if not ((hit_selected > 0) and (@state == State::Selected))
          if @attributes.process_event(boxlist, event, x0, y0, x, y, msg)
            if @core_box
              @bbox = @core_box.dup
              @attributes.each{|x| x.enlarge_bbox(@bbox) unless x == nil}
            end
            return true
          end
        end
      end  
    end
    ctrl = ((event.state & Gdk::Window::CONTROL_MASK) != 0)
    shift = ((event.state & Gdk::Window::SHIFT_MASK) != 0)
    old_state = @state
    if (msg == PEM::Hit_Select) or (msg ==  PEM::Drag_Select)
      if (msg == PEM::Hit_Select) and self.respond_to?(:special_hit_action) and (new_el = special_hit_action(objlist, x0, y0, x, y))
        return new_el
      end
      if msg == PEM::Hit_Select
        sel = @hoover
      else # msg ==  PEM::Drag_Select
        sel = (@bbox and Bounding::Box.new(x0, y0, x, y).include?(@bbox))
      end
      if sel
        if ctrl # toggle
          if @state == State::Visible
            @state = State::Selected
          else
            @state = State::Visible
          end
        else # add
          @state = State::Selected
        end
      elsif not (ctrl or shift)
        @state = State::Visible
      end
    elsif (msg == PEM::Delta_Move)# or (msg == PEM::First_Delta_Move)
      if @hoover and self.respond_to?(:special_move_action) and special_move_action(objlist, x0, y0, x, y)
        old = @bbox.dup()
        set_box()
        boxlist << old.join(@bbox)
        return true
      elsif ((hit_selected > 0) and (@state == State::Selected)) or ((hit_selected == 0) and @hoover)
        nb = @bbox.dup
        nb.enlarge(x, y)
        boxlist << nb
        self.translate(x, y)
        return true
      end
    end
    if (@state != old_state) or (@hoover != @old_hoover)
      boxlist << @bbox
    end
    return (@state != old_state)
  end


  def absorb(boxlist, event, x0, y0, x, y, msg)
#puts 'absorp'
    n = nil

if msg == PEM::Hoover_Select

#    if event.event_type == Gdk::Event::MOTION_NOTIFY
      #@x2 += x0
      #@x2 += y0
@x2, @y2 =  x0, y0
      boxlist << @bbox.enlarge(x, y)
    self.set_box()
elsif msg == PEM::Hit_Select

@absorbing = false

elsif msg == PEM::KEY_Delete

@absorbing = false



#return self


      #@bbox = Bounding::Box.new(@x1, @y1, @x2, @y2)
#    elsif (event.event_type == Gdk::Event::BUTTON_PRESS)
#      @x2, @y2 =  px, py
#      boxlist << @bbox.enlarge(px, py)
#      if ((@x1 - @x2) < 10) and ((@y1 - @y2) < 10)
#        @state = State::Deleted
##      else
#        @bbox = Bounding::Box.new(@x1, @y1, @x2, @y2)
#        n = self.dup
#        n.x1, n.y1 =  px, py
#        n.set_box
#        @state = State::Visible
#      end
    end
    return n
  end

  def check
    if not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n"
    elsif @color != Pet_Config::Colorindex_geda_net
      return " color index (#{@color}) should be #{NET_COLOR}\n" # warning
    elsif Pet.diagonal_line?(@x1, @y1, @x2, @y2)
      t = 'diagonal net'
    elsif not Pet.on_egrid?(@x1, @y1, @x2, @y2) 
      t = 'net segment is not on e-grid'
    elsif ((@x1 == @x2) and (@y1 == @y2))
      t = 'net segment of length 0'
    else
      t = ''
    end
    t <<  ", #{@x1}, #{@y1} -- #{@x2}, #{@y2}\n" unless t == ''
    return t
  end
  def to_s
    mytos @type, @x1, @y1, @x2, @y2, @color
  end
  def write(file)
    file.write to_s
  end
  def set_box()
@bbox = Bounding::Box.new(@x1, @y1, @x2, @y2).grow(32)

  end
  def translate(x, y)
    @x1 += x; @x2 += x; @y1 += y ;@y2 += y;
@bbox.translate(x, y)
  end


  def draw_junctions(cr, par, damage_list, draw_hoovered, draw_selected)
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
    if not @bbox.overlap_list?(damage_list) then return end

    [[@x1, @y1], [@x2, @y2]].each{|p|
      cons = cr.connections[[p[0], p[1]]]
      if cons > 2
        cr.set_color(*par[:color_geda_junction])
        cr.new_sub_path
        w = cr.clamped_line_width(par[:line_width_net])
        cr.arc(p[0], p[1], 0.5 * 3 * w, 0, 2 * Math::PI)
        cr.fill
      elsif cons == 1
        cr.set_color(*par[:color_geda_net_endpoint])
        w = 2 * cr.clamped_line_width(par[:line_width_net])
        cr.rectangle(p[0] - w, p[1] - w, 2 * w, 2 * w)
        cr.fill
      end
    }
  end


  def draw(cr, par, damage_list, draw_hoovered, draw_selected)

  if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
  if not @bbox.overlap_list?(damage_list) then return end

  #cr.rectangle(@bbox.x1, @bbox.y1, @bbox.x2 - @bbox.x1, @bbox.y2 - @bbox.y1)
  #cr.clip
    cr.set_line_cap(cr.g_cap[par[:net_end_cap]])
    cr.set_dash([], 0)

#if ((@hoover == true) or (@state > State::Visible))
#cr.highlight = true
#end
cr.set_color(*par[Pet_Config::CIM[@color]])

#puts @bbox.x1

xa, ya, xb, yb = @x1, @y1, @x2, @y2

if @y1 == @y2
xa = [@x1,  @bbox.x1].max

#puts 'ffffff', @x1,  @bbox.x1, xa


xb = [@x2,  @bbox.x2].min

elsif @x1 == @x2
ya = [@y1,  @bbox.y1].max


#puts 'eeeee', @y1,  @bbox.y1, ya


yb = [@y2,  @bbox.y2].min
#else

end

#puts 'ssss', @x1, @y1, @x2, @y2
#puts 'tttt', @bbox.x1, @bbox.y1, @bbox.x2, @bbox.y2

#puts 'rrrr', xa, ya, xb, yb




    cr.sharp_line(xa, ya, xb, yb, par[:line_width_net])

#    cr.sharp_line(x1, y1, x2, y2, par[:line_width_net])
#cr.highlight = false

=begin
    [[@x1, @y1], [@x2, @y2]].each{|p|
      cons = cr.connections[[p[0], p[1]]]
      if cons != 2
        if cons == 1
          cr.set_color(*par[:color_geda_net_endpoint])
        else
          cr.set_color(*par[:color_geda_junction])
        end
        cr.new_sub_path
        cr.arc(p[0], p[1], 10, 0, 2 * Math::PI)
        cr.fill_preserve
        cr.stroke
      end
    }
=end
  end
end

class Box < Element
  attr_accessor :x, :y, :width, :height, :color, :linewidth, :capstyle, :dashstyle, :dashlength, :dashspace, :filltype, :fillwidth, :angle1, :pitch1, :angle2, :pitch2
  def initialize
    super
    @type = BoxChar
    @x, @y, @width, @height, @color, @linewidth = 0, 0, 0, 0, 0, 0 # overwrite, or check() will complain
    @capstyle = GEDA::END_CAP[:ROUND]
    @dashstyle = GEDA::LINE_TYPE[:SOLID]
    @dashlength = -1
    @dashspace = -1
    @filltype = GEDA::FILLING[:HOLLOW]
    @fillwidth = -1
    @angle1 = -1
    @pitch1 = -1
    @angle2 = -1
    @pitch2 = -1
    self.set_box()
  end

  def set_box()
@bbox = Bounding::Box.new(@x1, @y1, @x1 + @width, @y1 + @height).grow(200)

  end

  def Box.start(x, y)
    b = Box.new
    b.x, b.y = x, y
    b.width = 0
    b.height = 0
    b.absorbing = true
    return b
  end

  def absorb(boxlist, event, x0, y0, x, y, msg)

puts 'msg', msg

    b = nil

if (event.event_type == Gdk::Event::Type::MOTION_NOTIFY)
@width = x0 - @x
@height = y0 - @y

#@x2, @y2 =  x0, y0 
      boxlist << @bbox.enlarge(x, y)
    self.set_box()

puts 'box', @bbox.x1, @bbox.y1, @bbox.x2, @bbox.y2


elsif (event.event_type == Gdk::Event::Type::BUTTON_PRESS)
#else
@absorbing = false

#elsif msg == PEM::KEY_Delete

#@absorbing = false




    end
    return b
  end

  def check
    if @linewidth < 0
      return "line width should not be negative (#{@linewidth})\n"
    elsif (@filltype == GEDA::FILLING[:MESH]) or (@filltype == GEDA::FILLING[:HATCH])
      if @pitch1 <= 0
        return "pitch1 for filling should be > 0(#{@pitch1})\n"
      elsif not (-360..360).include? @angle1
        return "angle1 for filling should be in the range -360..360 (#{@angle1})\n"
      end
      if @filltype == GEDA::FILLING[:MESH]
        if @pitch2 <= 0
          return "pitch2 for filling should be > 0(#{@pitch2})\n"
        elsif not (-360..360).include? @angle2
          return "angle2 for filling should be in the range -360..360 (#{@angle2})\n"
        elsif @angle1 == @angle2
          return " mesh with angle1 == angle2 (#{@angle1})\n" # warning, may be intended for different pitch
        end
      end
    elsif not GEDA::END_CAP.has_value? @capstyle
      return "unsupported cap style (#{@capstyle})\n"
    elsif not GEDA::LINE_TYPE.has_value? @dashstyle
      return "unsupported dash style (#{@dashstyle})\n"
    elsif (@dashstyle != GEDA::LINE_TYPE[:SOLID]) and (@dashspace <= 0)
      return "dashspace should be >= 0 (#{dashspace})\n" # we should query result of cairo stroke for other problems 
    elsif not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n" 
    elsif [@width, @height].min < MIN_STRUCTURE_SIZE # should we support negative @width, @height?
      return "very small rectangle (#{@width}, #{@height})\n"
    else
      return ''
    end
  end
  def to_s
    mytos @type, @x, @y, @width, @height, @color, @linewidth, @capstyle, @dashstyle, @dashlength, @dashspace, @filltype, @fillwidth, @angle1, @pitch1, @angle2, @pitch2
  end
  def write(file)
    file.write to_s
  end
  def set_box()
@bbox = Bounding::Box.new(@x, @y, @x + @width, @y + @height)
  end
  def translate(x, y)
    @x += x; @y += y;
@bbox.translate(x, y)

  end
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)


cr.set_color(*par[Pet_Config::CIM[@color]])
     if (@filltype == GEDA::FILLING[:MESH]) or (@filltype == GEDA::FILLING[:HATCH])
      cr.save
      cr.rectangle(@x, @y, @width, @height)
      cr.translate(@x + @width * 0.5, @y + @height * 0.5)
      cr.set_dash([], 0)
      cr.clamped_set_line_width(@fillwidth)
      z = Math::sqrt(@width ** 2 + @height ** 2) * 0.5 
      cr.rotate(@angle1 / 180.0 * Math::PI)
      p = - z
      while p < z do
        cr.move_to(-z, p)
        cr.line_to(z, p)
        p += @pitch1
      end
      cr.stroke
      if (@filltype == GEDA::FILLING[:MESH])
        cr.rotate((@angle2 - @angle1) / 180.0 * Math::PI)
        p = - z
        while p < z do
          cr.move_to(-z, p)
          cr.line_to(z, p)
          p += @pitch2
        end
        cr.stroke
      end
      cr.restore
    end
    if (@filltype == GEDA::FILLING[:FILL]) # we have no separate fill color
      cr.rectangle(@x, @y, @width, @height)
      cr.fill
    end
    cr.set_line_join(cr.g_join[@capstyle])
    cr.geda_set_dash(@dashstyle, @dashlength, @dashspace)
    cr.sharp_rect(@x, @y, @width, @height, @linewidth)
  end
end

class Circ < Element
  attr_accessor :x, :y, :radius, :color, :width, :capstyle, :dashstyle, :dashlength, :dashspace, :filltype, :fillwidth, :angle1, :pitch1, :angle2, :pitch2
  def initialize
    super
    @type = CircChar
    @x, @y, @radius, @color, @width = 0, 0, 0, 0, 0 # overwrite, or check() will complain
    @capstyle = GEDA::END_CAP[:ROUND]
    @dashstyle = GEDA::LINE_TYPE[:SOLID]
    @dashlength = -1
    @dashspace = -1
    @filltype = GEDA::FILLING[:HOLLOW]
    @fillwidth = -1
    @angle1 = -1
    @pitch1 = -1
    @angle2 = -1
    @pitch2 = -1
    @blur = 0
  end
  def check
    if @linewidth < 0
      return "line width should not be negative (#{@linewidth})\n"
    elsif (@filltype == GEDA::FILLING[:MESH]) or (@filltype == GEDA::FILLING[:HATCH])
      if @pitch1 <= 0
        return "pitch1 for filling should be > 0(#{@pitch1})\n"
      elsif not (-360..360).include? @angle1
        return "angle1 for filling should be in the range -360..360 (#{@angle1})\n"
      end
      if @filltype == GEDA::FILLING[:MESH]
        if @pitch2 <= 0
          return "pitch2 for filling should be > 0(#{@pitch2})\n"
        elsif not (-360..360).include? @angle2
          return "angle2 for filling should be in the range -360..360 (#{@angle2})\n"
        elsif @angle1 == @angle2
          return " mesh with angle1 == angle2 (#{@angle1})\n" # warning, may be intended for different pitch
        end
      end
    elsif not GEDA::END_CAP.has_value? @capstyle
      return "unsupported cap style (#{@capstyle})\n"
    elsif not GEDA::LINE_TYPE.has_value? @dashstyle
      return "unsupported dash style (#{@dashstyle})\n"
    elsif (@dashstyle != GEDA::LINE_TYPE[:SOLID]) and (@dashspace <= 0)
      return "dashspace should be >= 0 (#{dashspace})\n" # we should query result of cairo stroke for other problems 
    elsif not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n" 
    elsif @radius < MIN_STRUCTURE_SIZE
      return "very small circle (#{@radius})\n"
    else
      return ''
    end
  end
  def to_s
    mytos @type, @x, @y , @radius, @color, @width, @capstyle, @dashstyle, @dashlength, @dashspace, @filltype, @fillwidth, @angle1, @pitch1, @angle2, @pitch2
  end
  def write(file)
    file.write to_s
  end
  def set_box()
@bbox = Bounding::Box.new(@x - @radius, @y - @radius, @x + @radius, @y + @radius).grow(@blur)
  end
  def translate(x, y)
    @x += x; @y += y
@bbox.translate(x, y)
  end
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)
cr.set_color(*par[Pet_Config::CIM[@color]])
    if (@filltype == GEDA::FILLING[:MESH]) or (@filltype == GEDA::FILLING[:HATCH])
      cr.save
      cr.new_sub_path
      cr.arc(@x, @y, @radius, 0, 2 * Math::PI)
      #cr.clip
      cr.translate(@x, @y)
      cr.set_dash([], 0)
      cr.clamped_set_line_width(@fillwidth)
      z = @radius 
      cr.rotate(@angle1 / 180.0 * Math::PI)
      p = - z
      while p < z do
        cr.move_to(-z, p)
        cr.line_to(z, p)
        p += @pitch1
      end
      cr.stroke
      if (@filltype == GEDA::FILLING[:MESH])
        cr.rotate((@angle2 - @angle1) / 180.0 * Math::PI)
        p = - z
        while p < z do
          cr.move_to(-z, p)
          cr.line_to(z, p)
          p += @pitch2
        end
        cr.stroke
      end
      cr.restore
    end
    cr.set_line_join(cr.g_join[@capstyle])
    cr.geda_set_dash(@dashstyle, @dashlength, @dashspace)
    cr.clamped_set_line_width(@width)
@bbox = Bounding::Box.new(@x - @radius, @y - @radius, @x + @radius, @y + @radius).grow(cr.line_pollution(@width))
#    if @blur == 0
#      @blur = cr.line_pollution(@width)
#      @bbox.grow(@blur)
#    end
    cr.new_sub_path
    cr.arc(@x, @y, @radius, 0, 2 * Math::PI)
    if (@filltype == GEDA::FILLING[:FILL]) # we have no separate fill color
      cr.fill_preserve
    end
    cr.stroke
  end
end

class Arc < Element
end

class Path < Element
end



class Attr_Msg
  ID_NEW = 0
  ID_INHERITED = 1
  ID_REDEFINED = 2
  attr_accessor :name, :name_visible, :value, :value_visible, :history, :id
  def initialize(name, name_visible, value, value_visible, history, id)
    @name, @name_visible, @value, @value_visible, @history, @id = name, name_visible, value, value_visible, history, id
  end
end

GEDA_TEXT_INVISIBLE = 0
GEDA_TEXT_VISIBLE = 1
GEDA_TEXT_VIS_RANGE = GEDA_TEXT_INVISIBLE..GEDA_TEXT_VISIBLE
GEDA_TEXT_SHOW_NAME_VALUE = 0
GEDA_TEXT_SHOW_VALUE = 1
GEDA_TEXT_SHOW_NAME = 2
GEDA_TEXT_SHOW_RANGE = GEDA_TEXT_SHOW_NAME_VALUE..GEDA_TEXT_SHOW_NAME
TEXT_SIZE_DEFAULT = 10
TEXT_SIZE_MIN = 2
GEDA_TEXT_ALIGNMENT_RANGE = 0..8
# alignment
# 2 5 8
# 1 4 7
# 0 3 6
# x = -(a / 3) * 0.5 # origin transformation: cairo/pango top/left to gschem's alignment 
# y = -1 + (a % 3) * 0.5
# Attribute (name = value) or plain text
class Text < Element
  attr_accessor :x, :y, :color, :size, :visibility, :show_name_value, :angle, :alignment, :num_lines
  attr_accessor :lines
  def initialize
    super
    @type = TextChar
    @x, @y = 0, 0
    @color = Pet_Config::Colorindex_geda_text
    @size = TEXT_SIZE_DEFAULT
    @visibility = GEDA_TEXT_VISIBLE
    @show_name_value = GEDA_TEXT_SHOW_VALUE
    @angle = 0
    @alignment = 0
    @num_lines = 0
    @lines = Array.new
  end
  def check
    if not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n"
    elsif @size < TEXT_SIZE_MIN
      return "tiny text (#{@size})\n"
    elsif not GEDA_TEXT_SHOW_RANGE.include? @show_name_value
      return "show name/value is invalid (#{@show_name_value})\n"
    elsif not GEDA_TEXT_VIS_RANGE.include? @visibility
      return "visibility is a boolean 0/1 (#{@visibility})\n"
    elsif not GEDA_TEXT_ALIGNMENT_RANGE.include? @alignment
      return "alignment range is 0..8 (#{@alignment})\n"
    elsif not (-360..360).include? @angle
      return "angle should be in the range -360..360 (#{@angle})\n"
    elsif @num_lines <= 0 
      return "num lines should be > 0 (#{@num_lines})\n"
    elsif @num_lines != @lines.length 
      return "num lines (#{@num_lines}) != length of array (#{@lines.length})\n"
    else
      return ''
    end
  end
  def to_s
    mytos(@type, @x, @y, @color, @size, @visibility, @show_name_value, @angle, @alignment, @num_lines) + @lines.join("\n") + "\n"
  end
  def write(file)
    file.write to_s
  end

  def translate(x, y)
    @x += x; @y += y
  end
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
    if (@num_lines <= 0) or (@lines.length == 0) or (@visibility == GEDA_TEXT_INVISIBLE) then return end
    if @show_name_value == GEDA_TEXT_SHOW_NAME_VALUE
      d = @lines
    else
      d = Array.new
      @lines.each do |l| # each line ends with \n
        n, v = l.split('=')
        if v == nil
          d.push l
        elsif @show_name_value == GEDA_TEXT_SHOW_VALUE
          d.push v
        else
          d.push(n + "\n")
        end
      end
    end
    t = d.join
    overline = false
    l = t.length - 1 # last char is \n
    i, j = 0, 0
    tt = String.new
    o = Array.new
    while i < l do
      if (t[i, 1] == '\\') and (t[i + 1, 1] == '_')
        overline = (not overline)
        i += 2
      else
        tt << t[i]
        if overline
          o.push j
        end
        i += 1
        j += 1
      end
    end
    cr.save
    cr.translate(@x, @y)
    cr.rotate(@angle * Math::PI / 180.0)
    d = par[:text_mark_size] * 0.5
    if par[:text_mark_visible_always] and (d > 0) # or selected
w = cr.line_width_unscaled_user_min
cr.line_width_unscaled_user_min = 0
      cr.clamped_set_line_width(par[:text_mark_width])
cr.line_width_unscaled_user_min = w
      cr.set_line_cap(Cairo::LINE_CAP_ROUND)
      cr.set_dash([], 0)
      cr.move_to(-d, -d)
      cr.line_to(d, d)
      cr.move_to(-d, d)
      cr.line_to(d, -d)
      cr.stroke
    end
    cr.scale(1, -1) # reset our mirrored y axis, now we have the default again

#if ((@hoover == true) or (@state > State::Visible))
#cr.highlight = true
#end

cr.set_color(*par[Pet_Config::CIM[@color]])

#cr.highlight = false


    layout = cr.create_pango_layout
    layout.set_text(tt)
    desc = Pango::FontDescription.new(par[:text_font_name])
    desc.set_size(@size * par[:text_size_system_scale] * par[:text_size_user_scale] * Pango::SCALE)

desc.set_weight(400 + cr.text_shadow_fix)

    layout.set_font_description(desc)
    unless par[:text_field_transparent]
      al = Pango::AttrList.new
#      background_attr = Pango::AttrBackground.new(*cvalue[BACKGROUND_COLOR].map{|x| x * (2**16 - 1)})
      background_attr.start_index = 0 # Pango::ATTR_INDEX_FROM_TEXT_BEGINNING # Since: 1.24
      background_attr.end_index = -1 # Pango::ATTR_INDEX_TO_TEXT_END # Since: 1.24
      al.insert(background_attr)
      layout.set_attributes(al)
    end
    cr.update_pango_layout(layout)
    inc_rect, log_rect = layout.extents
    @bbox = Bounding::Box.new(@x, @y, @x + log_rect.width/ Pango::SCALE, @y + log_rect.height/ Pango::SCALE)
    cr.translate((-(@alignment / 3) * 0.5) * log_rect.width / Pango::SCALE, (-1 + (@alignment % 3) * 0.5) * log_rect.height / Pango::SCALE)
    cr.move_to(0, 0)
    cr.show_pango_layout(layout)
    i = o.length
    if i > 0
      cr.set_line_cap(Cairo::LINE_CAP_ROUND)
      cr.set_dash([], 0)
      cr.set_line_width(@size * par[:text_size_system_scale] * par[:text_size_user_scale] * 0.1)
      while i > 0
        i -= 1
        pos = layout.index_to_pos(o[i])
        cr.move_to(pos.x / Pango::SCALE, pos.y / Pango::SCALE)
        cr.rel_line_to(pos.width / Pango::SCALE, 0)
        cr.stroke
      end
    end
    cr.restore
#cr.rectangle(@bbox.x1, @bbox.y1, @bbox.x2 - @bbox.x1, @bbox.y2 - @bbox.y1)
#cr.stroke

  end
end

NORMAL_PIN = 0
BUS_PIN = 1 # unused
class Pin < Element
  attr_accessor :x1, :y1, :x2, :y2, :color, :pintype, :whichend
  attr_accessor :attributes
  def initialize
    super
    @type = PinChar
    @x1, @y1, @x2, @y2 = 0, 0, 0, 0
    @color = Pet_Config::Colorindex_geda_pin
    @pintype = NORMAL_PIN
    @whichend = 0
    @attributes = Pet_Object_List.new
  end

  def init_connections(h)
    h[[@x1, @y1]] += 1
  end

  def connect(px, py)
    if (@x1 - px)**2 + (@y1 - py)**2 < 50**2 then return [@x1, @y1] else return nil end
  end
  def check
    #@attributes.each{|a| if (a.class != String) or (a.count('=') != 1) return "invalid attribute (#{a})\n"}
    if not Pet_Config::COLOR_INDEX_RANGE.include? @color
      return "color index #{@color} out of range\n"
    elsif @color != Pet_Config::Colorindex_geda_pin
      return " color index (#{@color}) should be #{PIN_COLOR}\n" # warning
    elsif @pintype != 0
      return "pintype #{@pintype} should be NORMAL_PIN == 0\n"
    elsif (@whichend != 0) and (@whichend != 1)
      return "whichend #{@whichend} should be 0 or 1\n"
    elsif Pet.diagonal_line?(@x1, @y1, @x2, @y2)
      t = 'diagonal pin'
#    elsif ((@whichend == 0) and not on_egrid?(@x1, @y1)) or ((@whichend == 1) and not on_egrid?(@x2, @y2)) 
    elsif not Pet.on_egrid?(@x1, @y1) 
      t = 'active pin end is not on e-grid'
    elsif line_too_short?(@x1, @y1, @x2, @y2)
      t = 'very short pin'
    else
      t = ''
    end
    t <<  ", #{@x1}, #{@y1} -- #{@x2}, #{@y2}\n" unless t == ''
    return t
  end
  def attr_to_s
    if attributes.empty? then '' else "{\n" + @attributes.join("\n") + "}\n" end
  end
  def to_s
    if @whichend == 0
      mytos(@type, @x1, @y1, @x2, @y2, @color, @pintype, @whichend) + attr_to_s
    else
      mytos(@type, @x2, @y2, @x1, @y1, @color, @pintype, @whichend) + attr_to_s
    end
  end
  def write(file)
    file.write to_s
  end
  def set_box()
@bbox = Bounding::Box.new(@x1, @y1, @x2, @y2)
  end
  def translate(x, y)
    @x1 += x; @x2 += x; @y1 += y ;@y2 += y;
@bbox.translate(x, y)
  end
  def rotate(x, y, angle)
    @x1, @y1 = Pet.rtate(@x1, @y1, x, y, angle)
    @x2, @y2 = Pet.rtate(@x2, @y2, x, y, angle)
    set_box
  end

  def draw_junctions(cr, par, damage_list, draw_hoovered, draw_selected)
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
    if not @bbox.overlap_list?(damage_list) then return end
      if cr.connections[[@x1, @y1]] > 2
        cr.set_color(*par[:color_geda_junction])
        cr.new_sub_path
        w = cr.clamped_line_width(par[:line_width_net])
        cr.arc(@x1, @y1, 0.5 * 3 * w, 0, 2 * Math::PI)
        cr.fill
      end
  end

  def draw(cr, par, damage_list, draw_hoovered, draw_selected)
    cr.set_line_cap(cr.g_cap[par[:pin_end_cap]])
    cr.set_dash([], 0)
cr.set_color(*par[Pet_Config::CIM[@color]])
    cr.sharp_line(@x1, @y1, @x2, @y2, par[:line_width_pin])
# if pin not connected and mark_hot_pin_ends
#    cr.set_color(*cvalue[PIN_HOT_END_COLOR])
#    cr.set_color(*par[Pet_Config::DCV[@color]])

if cr.connections[[@x1, @y1]] == 1
cr.set_color(*par[:pin_hot_end_color])
    if @x1 == @x2 then
      if @y1 < @y2 then h = @y1 + 25 else h = @y1 - 25 end
      cr.sharp_line(@x1, @y1, @x2, h, par[:line_width_pin])
    else
      if @x1 < @x2 then h = @x1 + 25 else h = @x1 - 25 end
      cr.sharp_line(@x1, @y1, h, @y2, par[:line_width_pin])
    end
end
    @attributes.each{|x| x.draw(cr, par, damage_list, draw_hoovered, draw_selected)  unless x == nil}
  end
end


module State
  Deleted = 0
  Visible = 1
#  Hoover = 2
  Active = 3
  Selected = 4
  Moved = 5
Moving = Moved
  Absorbing = 6
  Pushed = 7
  Hit = 8
end

class EmbSym < Element
  attr_accessor :x, :y, :selectable, :angle, :mirror, :basename
  attr_accessor :components # pins, attributes and graphical elements
  attr_accessor :attributes 
  def initialize
    super
    @type = SymChar
    @x, @y = 0, 0
    @selectable = 0
    @angle = 0
    @mirror = 0
    @basename = nil
    @box_needs_fix = true
    @components = Array.new
    @attributes = Pet_Object_List.new
  end

  def get_attributes()
    list = Array.new
    @attributes.each{|el|
      msg = Attr_Msg.new(0,0,0,0,0,0)
      msg.name, msg.value = el.lines[0].split('=')
      list << msg
    }
    return list
  end

  def special_hit_action(px, py)
    @components.each{|c|
      if c.class == Pin
        if (cp = c.connect(px, py))
          return NetSegment.start(*cp)
        end
      end
    }
    return nil
  end

  def init_connections(h)
    @components.each{|x| x.init_connections(h) unless x == nil}
  end

  def check
    if @type != SymChar
      return "type should be SymChar (#{@SymChar}) but is #{@type}\n"
    elsif (@selectable != 0) and (@selectable != 1)
      return "selectable should be 0 or 1 (#{@selectable})\n"
    elsif not [0, 90, 180, 270].include? @angle
      return "angle should be 0, 90, 180 or 270 (#{@angle})\n"
    elsif (@mirror != 0) and (@mirror != 1)
      return "mirror should be 0 or 1 (#{@mirror})\n"
    else
      return ''
    end
  end
  def comp_to_s
    if @components.empty? then '' else "[\n" + @components.join("\n") + "]\n" end
  end
  def attr_to_s
   if @attributes.empty? then '' else "{\n" + @attributes.join("\n") + "}\n" end
  end
  def to_s
    mytos(@type, @x, @y, @selectable, @angle, @mirror, @basename) + comp_to_s + attr_to_s
  end
  def write(file)
    if self.class == EmbSym
      file.write to_s
    else
      file.write(mytos(@type, @x, @y, @selectable, @angle, @mirror, @basename) + attr_to_s)
    end
  end
  def set_box()
    @components.each{|x| x.set_box unless x == nil}
    @attributes.each{|x| x.set_box unless x == nil}
    b = Bounding::Box.new_ghost
    @components.each{|x| x.enlarge_bbox(b) unless x == nil}
    @core_box = b.dup
    @attributes.each{|x| x.enlarge_bbox(b) unless x == nil}
    @bx1, @by1, @bx2, @by2, = b.x1, b.y1, b.x2, b.y2
@bbox = b.grow(25)

  end


=begin
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
    if not @bbox.overlap_list?(damage_list) then return end
    if ((@hoover == true) or (@state > State::Visible))
      cr.highlight = true
    end
    @components.each{|x| x.draw(cr, par, damage_list, false, false) unless x == nil}
    @attributes.each{|x| x.draw(cr, par, damage_list, false, false)  unless x == nil}
    cr.highlight = false
    if @box_needs_fix == true
      @attributes.each{|at| at.enlarge_bbox(@bbox) unless at == nil} # temporary fix
      @box_needs_fix = false
    end
  end
=end

  def draw(cr, par, damage_list, draw_hoovered, draw_selected)

    if not @bbox.overlap_list?(damage_list) then return end
    @attributes.each{|x| x.draw(cr, par, damage_list, draw_hoovered, draw_selected)  unless x == nil}
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end

#    if ((@hoover == true) or (@state > State::Visible))
#      cr.highlight = true
#    end
    @components.each{|x| x.draw(cr, par, damage_list, false, false) unless x == nil}

#    cr.highlight = false
    if @box_needs_fix == true
      @attributes.each{|at| at.enlarge_bbox(@bbox) unless at == nil} # temporary fix
      @box_needs_fix = false
    end
  end

  def draw_junctions(cr, par, damage_list, draw_hoovered, draw_selected)
    if (draw_hoovered != @hoover) or (draw_selected != (@state == State::Selected)) then return end
    if not @bbox.overlap_list?(damage_list) then return end
#    @components.each{|x| x.draw_junctions(cr, par, damage_list, draw_hoovered, draw_selected)  unless x == nil}
    @components.each{|x| x.draw_junctions(cr, par, damage_list, false, false)  unless x == nil}
  end

  def translate(x, y)
    @x += x; @y += y
@bbox.translate(x, y)
@core_box.translate(x, y)
    @components.each{|el| el.translate(x, y) unless el == nil  }
    @attributes.each{|el| el.translate(x, y) unless el == nil}
  end
end

class Sym < EmbSym
  def draw(cr, par, damage_list, draw_hoovered, draw_selected)
    cr.save
    super(cr, par,  damage_list, draw_hoovered, draw_selected)
    cr.restore
  end
end

# special character indicating end of file
# NewLine() will set @ThisLine[0] to this character if end of file is reached
EOFC = ['~']

CoordinateRange = 1_000_000


class Schem
  attr_accessor :major_grid, :minor_grid, :active_grid, :pda, :main_window, :filename
  def initialize()
	  @filename = ''
    @CR = nil
    @pda = nil
    @major_grid, @minor_grid, @active_grid = 100, 100, 100
    @input_mode = Input_Mode.default
    @ObjectList = Pet_Object_List.new
    @state = PES::Hoovering
    @ActiveObject = nil
    @SymDirs = Pet_Config::DefaultSymDirs   
    @ThisLine = ''
    @Error=''
    @InputFile = nil
    @SymbolFile = nil
@Last_selected = nil
    @X_Range = Array.new
    @Y_Range = Array.new

#@bbox = Bounding::Box.new_ghost
@bbox = Bounding::Box.new(0, 0, 1000, 1000)

@log = Array.new
@dia = nil
  end

  def load_sym(name)

  end

  def set_dialog_widget(d)
    @dia = d
  end

  def set_input_mode(m)
    @input_mode = Input_Mode.const_get(m)
  end
  
  def NoInitColors(dev)
    if dev == 'PNG'
      @ColValPNG = DefColVal
    end  
  end

  def SetColor(dev, name, r, g, b)
    name = name.to_sym
    col = [r, g, b]
    if (col.max > 1.0) or (col.min < 0.0) then return false end
    if dev == 'PNG'
      if @ColValPNG.include?(name)
        @ColValPNG[name] = col
        return true
      else
        return false
      end
    else
      return false
    end
  end
  
  def ReadPreferences(filename)
  # read name, value
    if dev ==  'PNG' then end
  end



  # we should check for duplicates -- to be done
  def OpenSymFile(name)
    for base in @SymDirs
      n1, n2 = Dir.glob(File.join(base, '**', name))
      if n1
        @SymbolFile = File.open(n1, 'r')
        break
      end
    end
    return @SymbolFile
  end

  def NextLine()
    if @SymbolFile 
      if not (@ThisLine = @SymbolFile.gets)
        @SymbolFile.close
        @SymbolFile = nil
        @ThisLine = ']'
      end
    elsif not (@ThisLine = @InputFile.gets)
      @ThisLine = EOFC
    end
    puts @ThisLine
  end

  def Pin?() return @ThisLine[0..0] == 'P' end

  def Net?() return @ThisLine[0..0] == 'N' end

  def FirstIs(c) return @ThisLine[0..0] == c end
  
  def ProcessVersion()
    if match = Regexp.new(VersionPat).match(@ThisLine)
      el = Versio.new
      el.version = match[2]
      el.fileformat_version = match[3]
      NextLine()
      return el
    else
      @Error = 'Invalid File Version'
    end
  end
  
  def ProcessLine()
    if match = Regexp.new(LinePat).match(@ThisLine)
      el = Line.new
      el.x1 = match[2].to_i
      el.y1 = match[3].to_i
      el.x2 = match[4].to_i
      el.y2 = match[5].to_i
#el.x2, el.y2 = rotate(el.x2, el.y2, el.x1, el.y1, 10)
      el.color = match[6].to_i
      el.width = match[7].to_i
      el.capstyle = match[8].to_i
      el.dashstyle = match[9].to_i
      el.dashlength = match[10].to_i
      el.dashspace = match[11].to_i
      @Error = el.check
      NextLine()
      return el
    else
      @Error = 'Invalid Line'
    end
  end

  def ProcessNetSegment()
    if match = Regexp.new(NetSegPat).match(@ThisLine)
      el = NetSegment.new(0,0)
      el.x1 = match[2].to_i
      el.y1 = match[3].to_i
      el.x2 = match[4].to_i
      el.y2 = match[5].to_i
      el.color = match[6].to_i
      @Error = el.check
#@Error = ''
      NextLine()
      return el
    else
      @Error = 'Invalid Net'
    end
  end

  def ProcessBox()
    if match = Regexp.new(BoxPat).match(@ThisLine)
      el = Box.new
      el.x = match[2].to_i
      el.y = match[3].to_i
      el.width = match[4].to_i
      el.height = match[5].to_i
      el.color = match[6].to_i
      el.linewidth = match[7].to_i
      el.capstyle = match[8].to_i
      el.dashstyle = match[9].to_i
      el.dashlength = match[10].to_i
      el.dashspace = match[11].to_i
      el.filltype = match[12].to_i
      el.fillwidth = match[13].to_i
      el.angle1 = match[14].to_i
      el.pitch1 = match[15].to_i
      el.angle2 = match[16].to_i
      el.pitch2 = match[17].to_i
      NextLine()
      return el
    else
      @Error = 'Invalid Box'
    end
  end

  def ProcessCirc()
    if match = Regexp.new(CircPat).match(@ThisLine)
      el = Circ.new
      el.x = match[2].to_i
      el.y = match[3].to_i
      el.radius = match[4].to_i
      el.color = match[5].to_i
      el.width = match[6].to_i
      el.capstyle = match[7].to_i
      el.dashstyle = match[8].to_i
      el.dashlength = match[9].to_i
      el.dashspace = match[10].to_i
      el.filltype = match[11].to_i
      el.fillwidth = match[12].to_i
      el.angle1 = match[13].to_i
      el.pitch1 = match[14].to_i
      el.angle2 = match[15].to_i
      el.pitch2 = match[16].to_i
      NextLine()
      return el
    else
      @Error = 'Invalid Circle'
    end
  end

  def ProcessArc()
    NextLine()
  end
  
  def ProcessPath()
  end

  def ProcessText()
    if match = Regexp.new(TextPat).match(@ThisLine)
      el = Text.new
      el.x = match[2].to_i
      el.y = match[3].to_i
      el.color = match[4].to_i
      el.size = match[5].to_i
      el.visibility = match[6].to_i
      el.show_name_value = match[7].to_i
      el.angle = match[8].to_i
      el.alignment = match[9].to_i
      el.num_lines = match[10].to_i
      NextLine()
    else
      @Error = 'Text: Invalid start'
      return nil
    end
    el.lines = Array.new
    i = 0
    while i < el.num_lines
      el.lines.push @ThisLine
      NextLine()
      i += 1
    end

#    el.lines[0] = @ThisLine    
    
#    if match = Regexp.new(AttrPat).match(@ThisLine)
#      NextLine()
#    else
#      NextLine()
#    end

    return el
  end
  
  def ProcessAttributes
    if  FirstIs('{')
      NextLine()
      a = Pet_Object_List.new
      while true
        if FirstIs('}')
          NextLine()
          break
        else
          a.push(ProcessText())
        end
      end
      return a
    else
      @Error = 'Attr: Missing {'
      return nil
    end
  end
  
  def ProcessPin()
    if match = Regexp.new(PinPat).match(@ThisLine)
      el = Pin.new
      el.x1 = match[2].to_i
      el.y1 = match[3].to_i
      el.x2 = match[4].to_i
      el.y2 = match[5].to_i
      el.color = match[6].to_i
      el.pintype = match[7].to_i
      el.whichend = match[8].to_i
      if el.whichend != 0
        el.x1, el.x2 = el.x2, el.x1
        el.y1, el.y2 = el.y2, el.y1
      end
      NextLine()
    else
      @Error = 'Pin: Invalid start'
      return nil
    end
    if FirstIs('{')
      el.attributes = ProcessAttributes()
    end
    return el
  end

  def ScanSym()
    el = Sym.new
    while @ThisLine != EOFC
      if FirstIs(VersionChar)
        el.components.push(ProcessVersion())
      elsif FirstIs(LineChar)
        el.components.push(ProcessLine())
      elsif FirstIs(NetSegChar)
        el.components.push(ProcessNetSeg())
      elsif FirstIs(BoxChar)
        el.components.push(ProcessBox())
      elsif FirstIs(CircChar)
        el.components.push(ProcessCirc())
      elsif FirstIs(ArcChar)
        el.components.push(ProcessArc())
      elsif FirstIs(PathChar)
        el.components.push(ProcessPath())
      elsif FirstIs(TextChar)
        el.components.push(ProcessText())
      elsif FirstIs(PinChar)
        el.components.push(ProcessPin())
      else
        @Error = 'Symbol: Syntax error'
        return nil
      end
    end
#    if FirstIs('{')
#      el.attributes = ProcessAttributes()
#    end
#    if el.class == Sym
#      el.components.each{|o| o.set_box; o.translate(el.x, el.y); o.rotate(el.x, el.y, el.angle)}
#    end
    return el
  end



  def ProcessSym()
    if match = Regexp.new(EmbeddedSymPat).match(@ThisLine)
      el = EmbSym.new
      NextLine()
      if FirstIs('[')
        NextLine()
      else
        @Error = 'Symbol: Missing [ for embedded symbol'
        return nil
      end
    elsif match = Regexp.new(ExternSymPat).match(@ThisLine)
      el = Sym.new
      if OpenSymFile(match[7])
        @SymFileName = match[7]
        NextLine()
      else
        @Error = 'Symbol: File not found'
        return nil
      end
    else
      @Error = 'Symbol: Invalid start'
      return nil
    end

    el.x = match[2].to_i
    el.y = match[3].to_i
    el.selectable = match[4].to_i
    el.angle = match[5].to_i
    el.mirror = match[6].to_i
    el.basename = match[7]   
    while true
      if FirstIs(']')
        NextLine()
        break
      elsif FirstIs(VersionChar)
        el.components.push(ProcessVersion())
      elsif FirstIs(LineChar)
        el.components.push(ProcessLine())
      elsif FirstIs(NetSegChar)
        el.components.push(ProcessNetSeg())
      elsif FirstIs(BoxChar)
        el.components.push(ProcessBox())
      elsif FirstIs(CircChar)
        el.components.push(ProcessCirc())
      elsif FirstIs(ArcChar)
        el.components.push(ProcessArc())
      elsif FirstIs(PathChar)
        el.components.push(ProcessPath())
      elsif FirstIs(TextChar)
        el.components.push(ProcessText())
      elsif FirstIs(PinChar)
        el.components.push(ProcessPin())
      else
        @Error = 'Symbol: Syntax error'
        return nil
      end
    end
    if FirstIs('{')
      el.attributes = ProcessAttributes()
    end
    if el.class == Sym
      el.components.each{|o| o.set_box; o.translate(el.x, el.y); o.rotate(el.x, el.y, el.angle)}
    end
    return el
  end

  def ProcessSymFile(name)
    begin
      @SymFileName = ''
      @Error = ''
      @InputFile = File.open(name, 'r')
      NextLine()
      @Sym = ScanSym()
      if @Error != ''
        print 'Error in file ', name, ': ', @Error, "\n"
        print '-=> Line ', @InputFile.lineno, ': ', @ThisLine, "\n"
      else
@pda.set_cursor(Pet_Canvas::Cursor_Type::ADD_SYM)
@main_window.push_msg('Press LMB to place symbol...')
      end
      @InputFile.close
    rescue => e
      puts e.message
    end
  end

  def ProcessInputFile(name)
    begin
      @SymFileName = ''
      @Error = ''
      @InputFile = File.open(name, 'r')
      NextLine()
      while (@Error == '') and not FirstIs(EOFC)
        if FirstIs(VersionChar)
          ProcessVersion()
          #@ObjectList.push(ProcessVersion()) # missing bounding box can cause trouble
        elsif FirstIs(LineChar)
          @ObjectList.push(ProcessLine())
        elsif FirstIs(NetSegChar)
          @ObjectList.push(ProcessNetSegment())
        elsif FirstIs(BoxChar)
          @ObjectList.push(ProcessBox())
        elsif FirstIs(CircChar)
          @ObjectList.push(ProcessCirc())
        elsif FirstIs(ArcChar)
          @ObjectList.push(ProcessArc())
        elsif FirstIs(PathChar)
          @ObjectList .push(ProcessPath())
        elsif FirstIs(TextChar)
          @ObjectList.push(ProcessText())
        elsif FirstIs(SymChar)
          @ObjectList.push(ProcessSym())
        else
          @Error = 'Syntax error'
          break
        end
      end
# if @Error[0] == ' ' then only a warning...
# we should log it...
      if @Error == ''
        #Log::puts "#{name} loaded."
      else
        if @SymbolFile
          Log::print 'Error in file ', @SymFileName, ': ', @Error, "\n"
          Log::print '-=> Line ', @SymbolFile.lineno, ': ', @ThisLine, "\n"
          @SymbolFile.close
        else
          Log::print 'Error in file ', name, ': ', @Error, "\n"
          Log::print '-=> Line ', @InputFile.lineno, ': ', @ThisLine, "\n"
        end 
      end
      @InputFile.close
      @bbox = Bounding::Box.new_ghost
      @ObjectList.each{|o| o.set_box; o.enlarge_bbox(@bbox)}
    rescue => e
      Log::warn e.message
    end
    return !e && (@Error == '')
  end

  def write_to_context(cr, damage_list, new_background)
    conf = Pet_Config::get_default_config.get_conf(Pet_Config::SCR_S)
    cr.connections = Hash.new(0)
    @ObjectList.each{|x| x.init_connections(cr.connections)} # instead of this we should update connections hash whenever nets/pins are moved
    cr.connections.each_pair{|k, v|                          # at least suppress this for scrolling...
      if v == 1
        @ObjectList.each{|n|
          if n.class == NetSegment
            if ((n.x1 == k[0]) and (n.x1 == n.x2) and (((k[1] > n.y1) or (k[1] > n.y2)) and ((k[1] < n.y1) or (k[1] < n.y2)))) or
               ((n.y1 == k[1]) and (n.y1 == n.y2) and (((k[0] > n.x1) or (k[0] > n.x2)) and ((k[0] < n.x1) or (k[0] < n.x2))))
              cr.connections[k] += 2
            end
          end
        }
      end
    }
    major = @major_grid #conf[:grid_size_major]
    minor = @minor_grid #conf[:grid_size_minor]
    mac = conf[:color_geda_mesh_grid_major]
    mic = conf[:color_geda_mesh_grid_minor]
    if minor > major
      mic, mac = mac, mic
      minor, major = major, minor
    end
    if new_background or !cr.background_pattern
      @old_origin_x, @old_origin_y  = cr.device_to_user(0, 0)
      box = cr.bbox
      cr.push_group
      cr.set_source_rgb(conf[:color_geda_background][0..2]) # ignore alpha
      cr.paint
      cr.device_grid_major = cr.user_to_device_distance(major, 0)[0].round
      mesh_grid_minor_frac = 1 # 0..1 from config
      mesh_grid_major_frac = 1
      [[minor, mic, mesh_grid_minor_frac], [major, mac, mesh_grid_major_frac]].each{|grid, col, frac|
        if (grid > 0) and (cr.user_to_device_scale(grid) > 4)
          xi1 = box.x1.to_i / grid * grid
          xi2 = box.x2.to_i / grid * grid + grid
          yi1 = box.y1.to_i / grid * grid
          yi2 = box.y2.to_i / grid * grid + grid
          cr.set_line_cap(frac == 0 ? Cairo::LINE_CAP_ROUND : Cairo::LINE_CAP_BUTT)
          if frac == 1
            cr.set_dash([], 0)
          else
            l = grid * frac
            cr.set_dash([l, grid - l], l * 0.5)
          end
          cr.set_source_rgba(col)
          wu = cr.line_width_unscaled_user_min * cr.line_width_scale * cr.hair_line_scale
          w = cr.user_to_device_scale(wu)
          wd = w.round
          wu = cr.device_to_user_scale(wd) if w > 1
          if wd < 1 then wd = 1 end
          cr.set_line_width(wu)
          even = wd.even?
          skip_major = ((grid == minor) and (mesh_grid_minor_frac == 1) and (mesh_grid_major_frac == 1))
          Range.new(xi1, xi2).step(grid){|j|
            cr.faster_sharp_thin_line_v(yi1, yi2, j, even) unless skip_major and (j.modulo(major) == 0)
          }
          cr.stroke
          Range.new(yi1, yi2).step(grid){|j|
            cr.faster_sharp_thin_line_h(xi1, xi2, j, even) unless skip_major and (j.modulo(major) == 0)
          }
          cr.stroke
        end
      }
      if cr.background_pattern then cr.background_pattern.destroy end
      cr.background_pattern = cr.pop_group.surface
    else
      x, y = cr.device_to_user(0, 0)
      x = (x - @old_origin_x) % major
      y = (y - @old_origin_y) % major
      if x * 2 > major then x -= major end
      if y * 2 > major then y -= major end
      x, y = cr.user_to_device_distance(-x, -y)
      cr.background_pattern_offset_x, cr.background_pattern_offset_y = x.round, y.round
    end

    cr.set_operator(Cairo::OPERATOR_CLEAR)
    damage_list.each{|box|
    if box.x2 > box.x1 # why this test?
      cr.rectangle(box.x1, box.y1, box.x2 - box.x1, box.y2 - box.y1)
      cr.fill
    end
    }
    cr.set_operator(Cairo::OPERATOR_OVER)
    @ObjectList.each{|x| x.draw(cr, conf, damage_list, false, false) unless x == nil}
    @ObjectList.each{|x| x.draw_junctions(cr, conf, damage_list, false, false) unless x == nil}
    cr.push_group
    cr.soft = true
    shift = 0.2 * cr.device_to_user_line_width(cr.unscaled_user_to_device_line_width(0))
    1.upto(3) {|i|
      cr.translate(shift, -shift)
      cr.line_shadow_fix += (shift.fdiv(cr.line_width_scale))
      cr.text_shadow_fix += 100
      @ObjectList.each{|x| x.draw(cr, conf, damage_list, (i != 2),  (i != 1)) unless x == nil}
      @ObjectList.each{|x| x.draw_junctions(cr, conf, damage_list, (i != 2),  (i != 1)) unless x == nil}
    }
    cr.line_shadow_fix = 0
    cr.text_shadow_fix = 0
    cr.soft = false
    cr.set_source_rgba(0, 0, 0, 1)
    cr.set_operator(Cairo::OPERATOR_IN)
    cr.paint
    cr.pop_group_to_source
    cr.paint(0.9)

    cr.save
    cr.highlight = true
    shift *= 0.5
    1.upto(3) {|i|
      cr.translate(-shift, shift)
      cr.line_shadow_fix += (shift.fdiv(cr.line_width_scale))
      cr.text_shadow_fix += 100
      @ObjectList.each{|x| x.draw(cr, conf, damage_list, (i != 2),  (i != 1)) unless x == nil}
      @ObjectList.each{|x| x.draw_junctions(cr, conf, damage_list, (i != 2),  (i != 1)) unless x == nil}
    }
    cr.highlight = false
    cr.restore
    cr.line_shadow_fix = 0
    cr.text_shadow_fix = 0
  end

  def absorbed?(boxlist, event, px, py)
    if @ObjectList.empty? then return false end
    last = @ObjectList[-1]
    if last.state == State::Absorbing
      new = last.absorb(boxlist, event, px, py)
      if new != nil then @ObjectList << new end # delete_at(-1) if state==State::Deleted
      return true
    end
    return false
  end

#  def check_hit(px, py)
#    @ObjectList.check_hit(px, py)
#  end

  def hoovering?()
    @ObjectList.hoover
    #@ObjectList.hoovering?()
  end

#  def selected_element()
#    @ObjectList.selected_element()
#  end

  def process_event(boxlist, event, px, py, grid)

	puts 'pevent1'
	puts @state


    if @input_mode != Input_Mode::Smart # rectangle


        absorbed = @ObjectList.process_event(boxlist, event, px, py, px, py, nil)



    else


    if event.event_type == Gdk::Event::Type::KEY_PRESS
#puts 'iiiiliiiiiiiii'
puts Gdk::Keyval.to_name(event.keyval)
      if event.keyval == Gdk::Keyval::GDK_KEY_Delete
        absorbed = @ObjectList.process_event(boxlist, event, @px, @py, px, py, PEM::KEY_Delete)

      elsif event.keyval == Gdk::Keyval::GDK_KEY_Up

puts 'UP'
        dx = 0
        dy = 100
        absorbed = @ObjectList.process_event(boxlist, event, @px, @py, dx, dy, PEM::Delta_Move)
        @px += dx
        @py += dy


      elsif event.keyval == Gdk::Keyval::GDK_KEY_m


if @state == PES::Hoovering 
  @state = PES::Moved
      @px, @py = px, py
elsif @state == PES::Moved
@state = PES::Hoovering 
end

      end
    elsif (event.event_type == Gdk::Event::Type::BUTTON_PRESS)

	puts 'pevent2'
if @Sym
@Sym.x = px
@Sym.y = py
@Sym.selectable = 1

    #if el.class == Sym
      @Sym.components.each{|o| o.set_box; o.translate(px, py); o.rotate(px, py, @Sym.angle)}
@Sym.set_box
    #end
@ObjectList.push(@Sym)
boxlist << @Sym.bbox
@Sym = nil


@pda.set_cursor(Pet_Canvas::Cursor_Type::DEFAULT)
@main_window.pop_msg
else


      @px, @py = px, py
      if @ObjectList.hoover
        @state = PES::Hit
      else
        @state = PES::Dragging
      end
end
    elsif (event.event_type == Gdk::Event::Type::MOTION_NOTIFY)
      if (@state == PES::Hit) or (@state == PES::Moved)
        dx = (px - @px).to_i
        dy = (py - @py).to_i
        dx = dx.abs / grid * grid * (dx <=> 0)
        dy = dy.abs / grid * grid * (dy <=> 0)
        if (dx != 0) or (dy != 0)
          @state = PES::Moved
          absorbed = @ObjectList.process_event(boxlist, event, @px, @py, dx, dy, PEM::Delta_Move)
          @px += dx
          @py += dy
        end
      elsif @state == PES::Hoovering
        #dx = (px- @px).to_i
        #dy = (py - @py).to_i
        #dx = px.to_i.abs / grid * grid * (px <=> 0)
        #dy = py.to_i.abs / grid * grid * (py <=> 0)


        dx = (px + 0.5 * grid).to_i / grid * grid
        dy = (py + 0.5 * grid).to_i / grid * grid
        
puts 'hoover'

        absorbed = @ObjectList.process_event(boxlist, event, dx, dy, px, py, PEM::Hoover_Select)
      end
    elsif (event.event_type == Gdk::Event::Type::BUTTON_RELEASE)
      if @state == PES::Hit
        absorbed = @ObjectList.process_event(boxlist, event, @px, @py, px, py, PEM::Hit_Select)
      elsif @state == PES::Dragging


	puts 'pevent3'

        absorbed = @ObjectList.process_event(boxlist, event, @px, @py, px, py, PEM::Drag_Select)
      end
      @state = PES::Hoovering 
    end
#puts @ObjectList.selected_element()
sel = @ObjectList.selected_element()
if sel != @Last_selected

@dia.refresh(sel)
@Last_selected = sel
end
end
  end

  def check_bbox(x, y)
    #boxlist = Array.new
    @ObjectList.each{|el| if el.check_bbox(x, y) == true then return true end}
    return false
  end



  def write(name)
    begin
      file = File.open(name, 'w')
      @ObjectList.each{|x| x.write(file)}
      file.close
    rescue => e
      puts e.message
    end
  end

  def bbox()
    [@bbox.x1, @bbox.y1, @bbox.x2 - @bbox.x1, @bbox.y2 - @bbox.y1]
  end

end # Schem

def main
  if (ARGV[0] == nil) or (ARGV[0] == '-h') or (ARGV[0] == '--help')
    print Docu
  else
    s1 = Schem.new
    s1.ProcessInputFile(ARGV[0])
    #s1.testpng

s1.write_png('out.png', 1400,1200)

    s1.write('txt.txt')
  end
end

# Start processing after all functions are read
#main
end # Pet
